/*
 * Copyright (c) "Neo4j"
 * Neo4j Sweden AB [https://neo4j.com]
 *
 * This file is part of Neo4j.
 *
 * Neo4j is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package org.neo4j.io.mem;

import org.neo4j.memory.MemoryTracker;

/**
 * A MemoryAllocator is simple: it only allocates memory, until it is closed and frees it all in one go.
 */
public interface MemoryAllocator {
    static MemoryAllocator createAllocator(long expectedMemory, MemoryTracker memoryTracker) {
        return createAllocator(expectedMemory, null, memoryTracker);
    }

    static MemoryAllocator createAllocator(long expectedMemory, Long grabSize, MemoryTracker memoryTracker) {
        return new GrabAllocator(expectedMemory, grabSize, memoryTracker);
    }

    /**
     * @return The sum, in bytes, of all the memory currently allocating through this allocator.
     */
    long usedMemory();

    /**
     * @return The amount of available memory, in bytes.
     */
    long availableMemory();

    /**
     * Allocate a contiguous, aligned region of memory of the given size in bytes.
     * @param bytes the number of bytes to allocate.
     * @param alignment The byte multiple that the allocated pointers have to be aligned at.
     * @return A pointer to the allocated memory.
     * @throws OutOfMemoryError if the requested memory could not be allocated.
     */
    long allocateAligned(long bytes, long alignment);

    /**
     * Close all allocated resources and free all allocated memory.
     * Closing can happen by calling close explicitly or by GC as soon as allocator will become phantom reachable.
     * It's up to implementations to guarantee correctness in scenario when multiple attempts will be made to release allocator resources.
     * As soon as allocated resources will be cleaned any code that will try to access previously available memory will not gonna be able to do so.
     */
    void close();
}
