/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.io.fs;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileDescriptor;
import java.io.FileFilter;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.io.RandomAccessFile;
import java.io.UncheckedIOException;
import java.lang.reflect.Field;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.channels.SeekableByteChannel;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.CopyOption;
import java.nio.file.DirectoryNotEmptyException;
import java.nio.file.FileStore;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardCopyOption;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.util.Collections;
import java.util.Objects;
import java.util.Set;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.commons.lang3.SystemUtils;
import org.apache.commons.lang3.reflect.FieldUtils;
import org.neo4j.function.Predicates;
import org.neo4j.util.FeatureToggles;

public class FileUtils {
    private static final boolean PRINT_REFLECTION_EXCEPTIONS;
    private static final int NUMBER_OF_RETRIES = 5;
    private static final Field CHANNEL_FILE_DESCRIPTOR;
    private static final Field FILE_DESCRIPTOR_FIELD;

    private FileUtils() {
        throw new AssertionError();
    }

    static int getFileDescriptor(FileChannel fileChannel) {
        Objects.requireNonNull(fileChannel);
        try {
            if (FILE_DESCRIPTOR_FIELD == null || CHANNEL_FILE_DESCRIPTOR == null) {
                return -1;
            }
            FileDescriptor fileDescriptor = (FileDescriptor)CHANNEL_FILE_DESCRIPTOR.get(fileChannel);
            return FILE_DESCRIPTOR_FIELD.getInt(fileDescriptor);
        }
        catch (IllegalAccessException | IllegalArgumentException e) {
            if (PRINT_REFLECTION_EXCEPTIONS) {
                e.printStackTrace();
            }
            return -1;
        }
    }

    public static void deleteRecursively(File directory) throws IOException {
        if (!directory.exists()) {
            return;
        }
        Path path = directory.toPath();
        FileUtils.deletePathRecursively(path);
    }

    public static void deletePathRecursively(Path path) throws IOException {
        FileUtils.deletePathRecursively(path, Predicates.alwaysTrue());
    }

    public static long blockSize(File file) throws IOException {
        return Files.getFileStore(file.toPath()).getBlockSize();
    }

    public static void deletePathRecursively(Path path, final Predicate<Path> removeFilePredicate) throws IOException {
        Files.walkFileTree(path, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){
            private int skippedFiles;

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                if (removeFilePredicate.test(file)) {
                    FileUtils.deleteFile(file);
                } else {
                    ++this.skippedFiles;
                }
                return FileVisitResult.CONTINUE;
            }

            @Override
            public FileVisitResult postVisitDirectory(Path dir, IOException e) throws IOException {
                if (e != null) {
                    throw e;
                }
                try {
                    if (this.skippedFiles == 0) {
                        Files.delete(dir);
                        return FileVisitResult.CONTINUE;
                    }
                    if (this.isDirectoryEmpty(dir)) {
                        Files.delete(dir);
                    }
                    return FileVisitResult.CONTINUE;
                }
                catch (DirectoryNotEmptyException notEmpty) {
                    String reason = this.notEmptyReason(dir, notEmpty);
                    throw new IOException(notEmpty.getMessage() + ": " + reason, notEmpty);
                }
            }

            private boolean isDirectoryEmpty(Path dir) throws IOException {
                try (Stream<Path> list = Files.list(dir);){
                    boolean bl = list.noneMatch(Predicates.alwaysTrue());
                    return bl;
                }
            }

            private String notEmptyReason(Path dir, DirectoryNotEmptyException notEmpty) {
                String string;
                block8: {
                    Stream<Path> list = Files.list(dir);
                    try {
                        string = list.map(p -> String.valueOf(p.getFileName())).collect(Collectors.joining("', '", "'", "'."));
                        if (list == null) break block8;
                    }
                    catch (Throwable throwable) {
                        try {
                            if (list != null) {
                                try {
                                    list.close();
                                }
                                catch (Throwable throwable2) {
                                    throwable.addSuppressed(throwable2);
                                }
                            }
                            throw throwable;
                        }
                        catch (Exception e) {
                            notEmpty.addSuppressed(e);
                            return "(could not list directory: " + e.getMessage() + ")";
                        }
                    }
                    list.close();
                }
                return string;
            }
        });
    }

    public static boolean deleteFile(File file) {
        boolean deleted;
        if (!file.exists()) {
            return true;
        }
        if (file.isDirectory()) {
            File[] files = file.listFiles();
            if (files == null) {
                return false;
            }
            if (files.length > 0) {
                return false;
            }
        }
        int count = 0;
        do {
            if (deleted = file.delete()) continue;
            ++count;
            FileUtils.waitAndThenTriggerGC();
        } while (!deleted && count <= 5);
        return deleted;
    }

    public static void moveFile(File toMove, File target) throws IOException {
        if (!toMove.exists()) {
            throw new FileNotFoundException("Source file[" + toMove.getAbsolutePath() + "] not found");
        }
        if (target.exists()) {
            throw new IOException("Target file[" + target.getAbsolutePath() + "] already exists");
        }
        if (toMove.renameTo(target)) {
            return;
        }
        if (toMove.isDirectory()) {
            Files.createDirectories(target.toPath(), new FileAttribute[0]);
            FileUtils.copyRecursively(toMove, target);
            FileUtils.deleteRecursively(toMove);
        } else {
            FileUtils.copyFile(toMove, target);
            FileUtils.deleteFile(toMove);
        }
    }

    public static File moveFileToDirectory(File toMove, File targetDirectory) throws IOException {
        if (!targetDirectory.isDirectory()) {
            throw new IllegalArgumentException("Move target must be a directory, not " + targetDirectory);
        }
        File target = new File(targetDirectory, toMove.getName());
        FileUtils.moveFile(toMove, target);
        return target;
    }

    public static void copyFileToDirectory(File file, File targetDirectory) throws IOException {
        if (!targetDirectory.exists()) {
            Files.createDirectories(targetDirectory.toPath(), new FileAttribute[0]);
        }
        if (!targetDirectory.isDirectory()) {
            throw new IllegalArgumentException("Move target must be a directory, not " + targetDirectory);
        }
        File target = new File(targetDirectory, file.getName());
        FileUtils.copyFile(file, target);
    }

    public static void renameFile(File srcFile, File renameToFile, CopyOption ... copyOptions) throws IOException {
        Files.move(srcFile.toPath(), renameToFile.toPath(), copyOptions);
    }

    public static void truncateFile(SeekableByteChannel fileChannel, long position) throws IOException {
        FileUtils.windowsSafeIOOperation(() -> fileChannel.truncate(position));
    }

    public static void truncateFile(File file, long position) throws IOException {
        try (RandomAccessFile access = new RandomAccessFile(file, "rw");){
            FileUtils.truncateFile(access.getChannel(), position);
        }
    }

    private static void waitAndThenTriggerGC() {
        try {
            Thread.sleep(500L);
        }
        catch (InterruptedException ee) {
            Thread.interrupted();
        }
        System.gc();
    }

    public static String fixSeparatorsInPath(String path) {
        String fileSeparator = System.getProperty("file.separator");
        if ("\\".equals(fileSeparator)) {
            path = path.replace('/', '\\');
        } else if ("/".equals(fileSeparator)) {
            path = path.replace('\\', '/');
        }
        return path;
    }

    public static void copyFile(File srcFile, File dstFile) throws IOException {
        FileUtils.copyFile(srcFile, dstFile, StandardCopyOption.REPLACE_EXISTING);
    }

    public static void copyFile(File srcFile, File dstFile, CopyOption ... copyOptions) throws IOException {
        dstFile.getParentFile().mkdirs();
        Files.copy(srcFile.toPath(), dstFile.toPath(), copyOptions);
    }

    public static void copyRecursively(File fromDirectory, File toDirectory) throws IOException {
        FileUtils.copyRecursively(fromDirectory, toDirectory, null);
    }

    public static void copyRecursively(File fromDirectory, File toDirectory, FileFilter filter) throws IOException {
        File[] files = fromDirectory.listFiles(filter);
        if (files != null) {
            for (File fromFile : files) {
                File toFile = new File(toDirectory, fromFile.getName());
                if (fromFile.isDirectory()) {
                    Files.createDirectories(toFile.toPath(), new FileAttribute[0]);
                    FileUtils.copyRecursively(fromFile, toFile, filter);
                    continue;
                }
                FileUtils.copyFile(fromFile, toFile);
            }
        }
    }

    public static void writeToFile(File target, String text, boolean append) throws IOException {
        if (!target.exists()) {
            Files.createDirectories(target.getParentFile().toPath(), new FileAttribute[0]);
            target.createNewFile();
        }
        try (OutputStreamWriter out = new OutputStreamWriter((OutputStream)new FileOutputStream(target, append), StandardCharsets.UTF_8);){
            out.write(text);
        }
    }

    public static PrintWriter newFilePrintWriter(File file, Charset charset) throws FileNotFoundException {
        return new PrintWriter(new OutputStreamWriter((OutputStream)new FileOutputStream(file, true), charset));
    }

    public static File path(String root, String ... path) {
        return FileUtils.path(new File(root), path);
    }

    public static File path(File root, String ... path) {
        for (String part : path) {
            root = new File(root, part);
        }
        return root;
    }

    public static boolean highIODevice(Path pathOnDevice) {
        if (SystemUtils.IS_OS_LINUX) {
            try {
                FileStore fileStore = Files.getFileStore(pathOnDevice);
                String name = fileStore.name();
                if (name.equals("tmpfs") || name.equals("hugetlbfs")) {
                    return true;
                }
                if (name.startsWith("/dev/nvme")) {
                    return true;
                }
                Path device = Paths.get(name, new String[0]).toRealPath(new LinkOption[0]);
                Path deviceName = device.getName(device.getNameCount() - 1);
                Path rotational = FileUtils.rotationalPathFor(deviceName);
                if (Files.exists(rotational, new LinkOption[0])) {
                    return FileUtils.readFirstCharacter(rotational) == 48;
                }
                String namePart = deviceName.toString();
                int len = namePart.length();
                while (Character.isDigit(namePart.charAt(len - 1))) {
                    --len;
                }
                deviceName = Paths.get(namePart.substring(0, len), new String[0]);
                rotational = FileUtils.rotationalPathFor(deviceName);
                if (Files.exists(rotational, new LinkOption[0])) {
                    return FileUtils.readFirstCharacter(rotational) == 48;
                }
            }
            catch (Exception ignored) {
                return true;
            }
        }
        return true;
    }

    private static Path rotationalPathFor(Path deviceName) {
        return Paths.get("/sys/block", new String[0]).resolve(deviceName).resolve("queue").resolve("rotational");
    }

    private static int readFirstCharacter(Path file) throws IOException {
        try (InputStream in = Files.newInputStream(file, StandardOpenOption.READ);){
            int n = in.read();
            return n;
        }
    }

    public static File pathToFileAfterMove(File fromDir, File toDir, File fileToMove) {
        Path fromDirPath = fromDir.toPath();
        Path toDirPath = toDir.toPath();
        Path fileToMovePath = fileToMove.toPath();
        return FileUtils.pathToFileAfterMove(fromDirPath, toDirPath, fileToMovePath).toFile();
    }

    public static Path pathToFileAfterMove(Path fromDir, Path toDir, Path fileToMove) {
        if (!fileToMove.startsWith(fromDir) || fileToMove.equals(fromDir)) {
            throw new IllegalArgumentException("File " + fileToMove + " is not a sub path to dir " + fromDir);
        }
        return toDir.resolve(fromDir.relativize(fileToMove));
    }

    public static long countFilesInDirectoryPath(Path dir) throws IOException {
        try (Stream<Path> listing = Files.list(dir);){
            long l = listing.count();
            return l;
        }
    }

    public static void windowsSafeIOOperation(Operation operation) throws IOException {
        IOException storedIoe = null;
        for (int i = 0; i < 5; ++i) {
            try {
                operation.perform();
                return;
            }
            catch (IOException e) {
                storedIoe = e;
                FileUtils.waitAndThenTriggerGC();
                continue;
            }
        }
        throw (IOException)Objects.requireNonNull(storedIoe);
    }

    public static void readTextFile(File file, LineListener listener) throws IOException {
        try (BufferedReader reader = new BufferedReader(new FileReader(file));){
            String line;
            while ((line = reader.readLine()) != null) {
                listener.line(line);
            }
        }
    }

    private static void deleteFile(Path path) throws IOException {
        FileUtils.windowsSafeIOOperation(() -> Files.delete(path));
    }

    public static String relativePath(File baseDir, File storeFile) throws IOException {
        String prefix = baseDir.getCanonicalPath();
        String path = storeFile.getCanonicalPath();
        if (!path.startsWith(prefix)) {
            throw new FileNotFoundException();
        }
        if ((path = path.substring(prefix.length())).startsWith(File.separator)) {
            return path.substring(1);
        }
        return path;
    }

    public static File getCanonicalFile(File file) {
        try {
            File fileToResolve = file.exists() ? file.toPath().toRealPath(new LinkOption[0]).toFile() : file;
            return fileToResolve.getCanonicalFile();
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    public static void writeAll(FileChannel channel, ByteBuffer src, long position) throws IOException {
        int bytesWritten;
        long filePosition = position;
        long expectedEndPosition = filePosition + (long)src.limit() - (long)src.position();
        while ((filePosition += (long)(bytesWritten = channel.write(src, filePosition))) < expectedEndPosition) {
            if (bytesWritten > 0) continue;
            throw new IOException("Unable to write to disk, reported bytes written was " + bytesWritten);
        }
    }

    public static void writeAll(FileChannel channel, ByteBuffer src) throws IOException {
        int bytesWritten;
        long bytesToWrite = src.limit() - src.position();
        while ((bytesToWrite -= (long)(bytesWritten = channel.write(src))) > 0L) {
            if (bytesWritten > 0) continue;
            throw new IOException("Unable to write to disk, reported bytes written was " + bytesWritten);
        }
    }

    public static FileChannel open(Path path, Set<OpenOption> options) throws IOException {
        return FileChannel.open(path, options, new FileAttribute[0]);
    }

    public static InputStream openAsInputStream(Path path) throws IOException {
        return Files.newInputStream(path, StandardOpenOption.READ);
    }

    public static OutputStream openAsOutputStream(Path path, boolean append) throws IOException {
        OpenOption[] options = append ? new OpenOption[]{StandardOpenOption.CREATE, StandardOpenOption.WRITE, StandardOpenOption.APPEND} : new OpenOption[]{StandardOpenOption.CREATE, StandardOpenOption.WRITE};
        return Files.newOutputStream(path, options);
    }

    public static String getFileStoreType(File file) {
        try {
            return Files.getFileStore(file.toPath()).type();
        }
        catch (IOException e) {
            return "Unknown file store type: " + e.getMessage();
        }
    }

    public static void tryForceDirectory(File directory) throws IOException {
        if (!directory.exists()) {
            throw new NoSuchFileException(String.format("The directory %s does not exist!", directory.getAbsolutePath()));
        }
        if (!directory.isDirectory()) {
            throw new IllegalArgumentException(String.format("The path %s must refer to a directory!", directory.getAbsolutePath()));
        }
        if (SystemUtils.IS_OS_WINDOWS) {
            return;
        }
        FileChannel directoryChannel = FileChannel.open(directory.toPath(), Collections.singleton(StandardOpenOption.READ), new FileAttribute[0]);
        directoryChannel.force(true);
    }

    static {
        Field fileDescriptorField;
        Field channelFileDescriptor;
        block2: {
            PRINT_REFLECTION_EXCEPTIONS = FeatureToggles.flag(FileUtils.class, (String)"printReflectionExceptions", (boolean)false);
            channelFileDescriptor = null;
            fileDescriptorField = null;
            try {
                Class<?> fileChannelClass = Class.forName("sun.nio.ch.FileChannelImpl");
                channelFileDescriptor = Objects.requireNonNull(FieldUtils.getDeclaredField(fileChannelClass, (String)"fd", (boolean)true));
                fileDescriptorField = FieldUtils.getDeclaredField(FileDescriptor.class, (String)"fd", (boolean)true);
            }
            catch (Exception e) {
                if (!PRINT_REFLECTION_EXCEPTIONS) break block2;
                e.printStackTrace();
            }
        }
        CHANNEL_FILE_DESCRIPTOR = channelFileDescriptor;
        FILE_DESCRIPTOR_FIELD = fileDescriptorField;
    }

    public static interface LineListener {
        public void line(String var1);
    }

    public static interface Operation {
        public void perform() throws IOException;
    }
}

