/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.io.pagecache.impl.muninn;

import java.io.File;
import java.io.IOException;
import java.util.concurrent.Executor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.LockSupport;
import org.neo4j.io.fs.FileUtils;
import org.neo4j.io.pagecache.PageCache;
import org.neo4j.io.pagecache.PageSwapperFactory;
import org.neo4j.io.pagecache.PagedFile;
import org.neo4j.io.pagecache.impl.muninn.BackgroundThreadExecutor;
import org.neo4j.io.pagecache.impl.muninn.CursorPool;
import org.neo4j.io.pagecache.impl.muninn.EvictionTask;
import org.neo4j.io.pagecache.impl.muninn.FileMapping;
import org.neo4j.io.pagecache.impl.muninn.FlushTask;
import org.neo4j.io.pagecache.impl.muninn.FreePage;
import org.neo4j.io.pagecache.impl.muninn.FreePageWaiter;
import org.neo4j.io.pagecache.impl.muninn.MemoryReleaser;
import org.neo4j.io.pagecache.impl.muninn.MuninnPage;
import org.neo4j.io.pagecache.impl.muninn.MuninnPagedFile;
import org.neo4j.io.pagecache.tracing.EvictionEvent;
import org.neo4j.io.pagecache.tracing.EvictionRunEvent;
import org.neo4j.io.pagecache.tracing.FlushEventOpportunity;
import org.neo4j.io.pagecache.tracing.MajorFlushEvent;
import org.neo4j.io.pagecache.tracing.PageCacheTracer;
import org.neo4j.io.pagecache.tracing.PageFaultEvent;
import org.neo4j.unsafe.impl.internal.dragons.UnsafeUtil;

public class MuninnPageCache
implements PageCache {
    private static final int pageFaultSpinCount = Integer.getInteger("org.neo4j.io.pagecache.impl.muninn.pageFaultSpinCount", FileUtils.OS_IS_WINDOWS ? 10 : 1000);
    private static final int pagesToKeepFree = Integer.getInteger("org.neo4j.io.pagecache.impl.muninn.pagesToKeepFree", 30);
    private static final long backgroundFlushSleepDebtThreshold = Long.getLong("org.neo4j.io.pagecache.impl.muninn.backgroundFlushSleepDebtThreshold", 10L);
    private static final double backgroundFlushIoRatio = MuninnPageCache.getDouble("org.neo4j.io.pagecache.impl.muninn.backgroundFlushIoRatio", 0.1);
    private static final long backgroundFlushBusyBreak = Long.getLong("org.neo4j.io.pagecache.impl.muninn.backgroundFlushBusyBreak", 100L);
    private static final long backgroundFlushMediumBreak = Long.getLong("org.neo4j.io.pagecache.impl.muninn.backgroundFlushMediumBreak", 200L);
    private static final long backgroundFlushLongBreak = Long.getLong("org.neo4j.io.pagecache.impl.muninn.backgroundFlushLongBreak", 1000L);
    private static final IOException oomException = new IOException("OutOfMemoryError encountered in the page cache background eviction thread");
    private static final long freelistOffset = UnsafeUtil.getFieldOffset(MuninnPageCache.class, (String)"freelist");
    private static final FreePageWaiter shutdownSignal = new FreePageWaiter();
    private static final AtomicInteger pageCacheIdCounter = new AtomicInteger();
    private static final Executor backgroundThreadExecutor = BackgroundThreadExecutor.INSTANCE;
    private final int pageCacheId;
    private final PageSwapperFactory swapperFactory;
    private final int cachePageSize;
    private final int keepFree;
    private final CursorPool cursorPool;
    private final PageCacheTracer tracer;
    private final MuninnPage[] pages;
    private final AtomicInteger backgroundFlushPauseRequests;
    private volatile Object freelist;
    private volatile FileMapping mappedFiles;
    private volatile Thread evictionThread;
    private volatile IOException evictorException;
    private volatile Thread flushThread;
    private volatile boolean closed;
    private boolean threadsInitialised;
    private long sleepDebtNanos;
    private boolean printExceptionsOnClose;

    private static double getDouble(String property, double def) {
        try {
            return Double.parseDouble(System.getProperty(property));
        }
        catch (Exception e) {
            return def;
        }
    }

    public MuninnPageCache(PageSwapperFactory swapperFactory, int maxPages, int cachePageSize, PageCacheTracer tracer) {
        MuninnPageCache.verifyHacks();
        MuninnPageCache.verifyCachePageSizeIsPowerOfTwo(cachePageSize);
        MuninnPageCache.verifyMinimumPageCount(maxPages, cachePageSize);
        this.pageCacheId = pageCacheIdCounter.incrementAndGet();
        this.swapperFactory = swapperFactory;
        this.cachePageSize = cachePageSize;
        this.keepFree = Math.min(pagesToKeepFree, maxPages / 2);
        this.cursorPool = new CursorPool();
        this.tracer = tracer;
        this.pages = new MuninnPage[maxPages];
        this.backgroundFlushPauseRequests = new AtomicInteger();
        this.printExceptionsOnClose = true;
        MemoryReleaser memoryReleaser = new MemoryReleaser(maxPages);
        Object pageList = null;
        int pageIndex = maxPages;
        while (pageIndex-- > 0) {
            FreePage freePage;
            MuninnPage page;
            this.pages[pageIndex] = page = new MuninnPage(cachePageSize, memoryReleaser);
            if (pageList == null) {
                freePage = new FreePage(page);
                freePage.setNext(null);
                pageList = freePage;
                continue;
            }
            if (pageList instanceof FreePage && ((FreePage)pageList).count < this.keepFree) {
                freePage = new FreePage(page);
                freePage.setNext((FreePage)pageList);
                pageList = freePage;
                continue;
            }
            page.nextFree = pageList;
            pageList = page;
        }
        UnsafeUtil.putObjectVolatile((Object)this, (long)freelistOffset, pageList);
    }

    private static void verifyHacks() {
        UnsafeUtil.assertHasUnsafe();
    }

    private static void verifyCachePageSizeIsPowerOfTwo(int cachePageSize) {
        int exponent = 31 - Integer.numberOfLeadingZeros(cachePageSize);
        if (1 << exponent != cachePageSize) {
            throw new IllegalArgumentException("Cache page size must be a power of two, but was " + cachePageSize);
        }
    }

    private static void verifyMinimumPageCount(int maxPages, int cachePageSize) {
        int minimumPageCount = 2;
        if (maxPages < minimumPageCount) {
            throw new IllegalArgumentException(String.format("Page cache must have at least %s pages (%s bytes of memory), but was given %s pages.", minimumPageCount, minimumPageCount * cachePageSize, maxPages));
        }
    }

    @Override
    public synchronized PagedFile map(File file, int filePageSize) throws IOException {
        this.assertHealthy();
        this.ensureThreadsInitialised();
        if (filePageSize > this.cachePageSize) {
            throw new IllegalArgumentException("Cannot map files with a filePageSize (" + filePageSize + ") that is greater than the cachePageSize (" + this.cachePageSize + ")");
        }
        FileMapping current = this.mappedFiles;
        while (current != null) {
            if (current.file.equals(file)) {
                MuninnPagedFile pagedFile = current.pagedFile;
                if (pagedFile.pageSize() != filePageSize) {
                    String msg = "Cannot map file " + file + " with " + "filePageSize " + filePageSize + " bytes, " + "because it has already been mapped with a " + "filePageSize of " + pagedFile.pageSize() + " bytes.";
                    throw new IllegalArgumentException(msg);
                }
                pagedFile.incrementRefCount();
                return pagedFile;
            }
            current = current.next;
        }
        MuninnPagedFile pagedFile = new MuninnPagedFile(file, this, filePageSize, this.swapperFactory, this.cursorPool, this.tracer);
        pagedFile.incrementRefCount();
        current = new FileMapping(file, pagedFile);
        current.next = this.mappedFiles;
        this.mappedFiles = current;
        this.tracer.mappedFile(file);
        return pagedFile;
    }

    private void ensureThreadsInitialised() throws IOException {
        if (this.threadsInitialised) {
            return;
        }
        this.threadsInitialised = true;
        try {
            backgroundThreadExecutor.execute(new EvictionTask(this));
            backgroundThreadExecutor.execute(new FlushTask(this));
        }
        catch (Exception e) {
            IOException exception = new IOException(e);
            try {
                this.close();
            }
            catch (IOException closeException) {
                exception.addSuppressed(closeException);
            }
            throw exception;
        }
    }

    synchronized void unmap(MuninnPagedFile file) {
        if (file.decrementRefCount()) {
            FileMapping prev = null;
            FileMapping current = this.mappedFiles;
            while (current != null) {
                if (current.pagedFile == file) {
                    if (prev == null) {
                        this.mappedFiles = current.next;
                    } else {
                        prev.next = current.next;
                    }
                    this.tracer.unmappedFile(current.file);
                    this.flushAndCloseWithoutFail(file);
                    break;
                }
                prev = current;
                current = current.next;
            }
        }
    }

    private void flushAndCloseWithoutFail(MuninnPagedFile file) {
        boolean flushedAndClosed = false;
        boolean printedFirstException = false;
        do {
            try {
                file.flushAndForce();
                file.closeSwapper();
                flushedAndClosed = true;
            }
            catch (IOException e) {
                if (!this.printExceptionsOnClose || printedFirstException) continue;
                printedFirstException = true;
                try {
                    e.printStackTrace();
                }
                catch (Exception exception) {
                    // empty catch block
                }
            }
        } while (!flushedAndClosed);
    }

    public void setPrintExceptionsOnClose(boolean enabled) {
        this.printExceptionsOnClose = enabled;
    }

    @Override
    public synchronized void flushAndForce() throws IOException {
        this.assertNotClosed();
        this.flushAllPages();
        this.clearEvictorException();
    }

    private void flushAllPages() throws IOException {
        try (MajorFlushEvent cacheFlush = this.tracer.beginCacheFlush();){
            FlushEventOpportunity flushOpportunity = cacheFlush.flushEventOpportunity();
            FileMapping fileMapping = this.mappedFiles;
            while (fileMapping != null) {
                fileMapping.pagedFile.flushAndForceInternal(flushOpportunity);
                fileMapping = fileMapping.next;
            }
        }
    }

    @Override
    public synchronized void close() throws IOException {
        if (this.closed) {
            return;
        }
        FileMapping files = this.mappedFiles;
        if (files != null) {
            StringBuilder msg = new StringBuilder("Cannot close the PageCache while files are still mapped:");
            while (files != null) {
                int refCount = files.pagedFile.getRefCount();
                msg.append("\n\t");
                msg.append(files.file.getName());
                msg.append(" (").append(refCount);
                msg.append(refCount == 1 ? " mapping)" : " mappings)");
                files = files.next;
            }
            throw new IllegalStateException(msg.toString());
        }
        this.closed = true;
        for (int i = 0; i < this.pages.length; ++i) {
            this.pages[i] = null;
        }
        this.interrupt(this.evictionThread);
        this.evictionThread = null;
        this.interrupt(this.flushThread);
        this.flushThread = null;
    }

    private void interrupt(Thread thread) {
        if (thread != null) {
            thread.interrupt();
        }
    }

    protected void finalize() throws Throwable {
        this.close();
        super.finalize();
    }

    private void assertHealthy() throws IOException {
        this.assertNotClosed();
        IOException exception = this.evictorException;
        if (exception != null) {
            throw new IOException("Exception in the page eviction thread", exception);
        }
    }

    private void assertNotClosed() {
        if (this.closed) {
            throw new IllegalStateException("The PageCache has been shut down");
        }
    }

    @Override
    public int pageSize() {
        return this.cachePageSize;
    }

    @Override
    public int maxCachedPages() {
        return this.pages.length;
    }

    int getPageCacheId() {
        return this.pageCacheId;
    }

    MuninnPage grabFreePage(PageFaultEvent faultEvent) throws IOException {
        FreePageWaiter waiter = null;
        int iterationCount = 0;
        boolean shouldUnparkInSpin = true;
        while (true) {
            this.assertHealthy();
            Object current = this.getFreelistHead();
            if (current == null && ++iterationCount > pageFaultSpinCount) {
                waiter = waiter == null ? new FreePageWaiter() : waiter;
                waiter.next = null;
                if (this.compareAndSetFreelistHead(null, waiter)) {
                    this.unparkEvictor();
                    faultEvent.setParked(true);
                    return waiter.park(this);
                }
            } else {
                if (current == null) {
                    if (!shouldUnparkInSpin) continue;
                    this.unparkEvictor();
                    shouldUnparkInSpin = false;
                    continue;
                }
                if (current instanceof MuninnPage) {
                    MuninnPage page = (MuninnPage)current;
                    if (this.compareAndSetFreelistHead(page, page.nextFree)) {
                        return page;
                    }
                } else if (current instanceof FreePage) {
                    FreePage freePage = (FreePage)current;
                    if (this.compareAndSetFreelistHead(freePage, freePage.next)) {
                        return freePage.page;
                    }
                } else if (current instanceof FreePageWaiter) {
                    if (current == shutdownSignal) {
                        throw new IllegalStateException("The PageCache has been shut down");
                    }
                    waiter = waiter == null ? new FreePageWaiter() : waiter;
                    waiter.next = (FreePageWaiter)current;
                    if (this.compareAndSetFreelistHead(current, waiter)) {
                        this.unparkEvictor();
                        faultEvent.setParked(true);
                        return waiter.park(this);
                    }
                }
            }
            this.unparkEvictor();
        }
    }

    private void unparkEvictor() {
        LockSupport.unpark(this.evictionThread);
    }

    private Object getFreelistHead() {
        return UnsafeUtil.getObjectVolatile((Object)this, (long)freelistOffset);
    }

    private boolean compareAndSetFreelistHead(Object expected, Object update) {
        return UnsafeUtil.compareAndSwapObject((Object)this, (long)freelistOffset, (Object)expected, (Object)update);
    }

    private Object getAndSetFreelistHead(Object newFreelistHead) {
        return UnsafeUtil.getAndSetObject((Object)this, (long)freelistOffset, (Object)newFreelistHead);
    }

    void continuouslySweepPages() {
        this.evictionThread = Thread.currentThread();
        int clockArm = 0;
        while (!Thread.interrupted()) {
            int pageCountToEvict = this.parkUntilEvictionRequired(this.keepFree);
            EvictionRunEvent evictionRunEvent = this.tracer.beginPageEvictions(pageCountToEvict);
            Throwable throwable = null;
            try {
                clockArm = this.evictPages(pageCountToEvict, clockArm, evictionRunEvent);
            }
            catch (Throwable throwable2) {
                throwable = throwable2;
                throw throwable2;
            }
            finally {
                if (evictionRunEvent == null) continue;
                if (throwable != null) {
                    try {
                        evictionRunEvent.close();
                    }
                    catch (Throwable x2) {
                        throwable.addSuppressed(x2);
                    }
                    continue;
                }
                evictionRunEvent.close();
            }
        }
        Object freelistHead = this.getAndSetFreelistHead(shutdownSignal);
        if (freelistHead instanceof FreePageWaiter) {
            FreePageWaiter waiters = (FreePageWaiter)freelistHead;
            this.interruptAllWaiters(waiters);
        }
    }

    private void interruptAllWaiters(FreePageWaiter waiters) {
        while (waiters != null) {
            waiters.unparkInterrupt();
            waiters = waiters.next;
        }
    }

    private int parkUntilEvictionRequired(int keepFree) {
        long parkNanos = TimeUnit.MILLISECONDS.toNanos(10L);
        while (true) {
            LockSupport.parkNanos(parkNanos);
            if (Thread.currentThread().isInterrupted() || this.closed) {
                return 0;
            }
            Object freelistHead = this.getFreelistHead();
            if (freelistHead instanceof FreePage) {
                int availablePages = ((FreePage)freelistHead).count;
                if (availablePages >= keepFree) continue;
                return keepFree - availablePages;
            }
            if (freelistHead instanceof FreePageWaiter) break;
        }
        return keepFree;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    int evictPages(int pageCountToEvict, int clockArm, EvictionRunEvent evictionRunEvent) {
        FreePageWaiter waiters = this.grabFreePageWaitersIfAny();
        Thread currentThread = Thread.currentThread();
        while (!(pageCountToEvict <= 0 && waiters == null || currentThread.isInterrupted())) {
            long stamp;
            MuninnPage page;
            if (clockArm == this.pages.length) {
                clockArm = 0;
            }
            if ((page = this.pages[clockArm]) == null) {
                currentThread.interrupt();
                this.interruptAllWaiters(waiters);
                return 0;
            }
            if (page.isLoaded() && page.decrementUsage() && (stamp = page.tryWriteLock()) != 0L) {
                boolean pageEvicted;
                --pageCountToEvict;
                try (EvictionEvent evictionEvent = evictionRunEvent.beginEviction();){
                    pageEvicted = this.evictPage(page, evictionEvent);
                }
                finally {
                    page.unlockWrite(stamp);
                }
                if (pageEvicted) {
                    if (waiters != null) {
                        waiters.unpark(page);
                        waiters = waiters.next;
                    } else {
                        FreePageWaiter waiter;
                        Object nextListHead;
                        Object current;
                        FreePage freePage = null;
                        do {
                            waiter = null;
                            current = this.getFreelistHead();
                            if (current == null || current instanceof FreePage) {
                                freePage = freePage == null ? new FreePage(page) : freePage;
                                freePage.setNext((FreePage)current);
                                nextListHead = freePage;
                                continue;
                            }
                            assert (current instanceof FreePageWaiter) : "Unexpected link type: " + current;
                            waiter = (FreePageWaiter)current;
                            nextListHead = waiter.next;
                        } while (!this.compareAndSetFreelistHead(current, nextListHead));
                        if (waiter != null) {
                            waiter.unpark(page);
                        }
                    }
                } else if (waiters != null && this.evictorException != null) {
                    waiters.unparkException(this.evictorException);
                    waiters = waiters.next;
                }
            }
            ++clockArm;
        }
        this.interruptAllWaiters(waiters);
        return clockArm;
    }

    private boolean evictPage(MuninnPage page, EvictionEvent evictionEvent) {
        try {
            page.evict(evictionEvent);
            this.clearEvictorException();
            return true;
        }
        catch (IOException ioException) {
            this.evictorException = ioException;
            evictionEvent.threwException(ioException);
        }
        catch (OutOfMemoryError ignore) {
            this.evictorException = oomException;
            evictionEvent.threwException(oomException);
        }
        catch (Throwable throwable) {
            this.evictorException = new IOException("Eviction thread encountered a problem", throwable);
            evictionEvent.threwException(this.evictorException);
        }
        return false;
    }

    private FreePageWaiter grabFreePageWaitersIfAny() {
        Object freelistHead = this.getFreelistHead();
        if (freelistHead instanceof FreePageWaiter) {
            FreePageWaiter waiters = (FreePageWaiter)this.getAndSetFreelistHead(null);
            return this.reverse(waiters);
        }
        return null;
    }

    private FreePageWaiter reverse(FreePageWaiter waiters) {
        FreePageWaiter result = null;
        while (waiters != null) {
            FreePageWaiter tail = waiters.next;
            waiters.next = result;
            result = waiters;
            waiters = tail;
        }
        return result;
    }

    private void clearEvictorException() {
        this.evictorException = null;
    }

    void pauseBackgroundFlushTask() {
        this.backgroundFlushPauseRequests.getAndIncrement();
    }

    void unpauseBackgroundFlushTask() {
        this.backgroundFlushPauseRequests.getAndDecrement();
        LockSupport.unpark(this.flushThread);
    }

    private void checkBackgroundFlushPause() {
        while (this.backgroundFlushPauseRequests.get() > 0) {
            LockSupport.parkNanos(TimeUnit.MILLISECONDS.toNanos(10L));
        }
    }

    void continuouslyFlushPages() {
        Thread thread;
        this.flushThread = thread = Thread.currentThread();
        while (!thread.isInterrupted()) {
            long iterationSleepMillis = this.flushAtIORatio(backgroundFlushIoRatio);
            if (iterationSleepMillis <= 0L) continue;
            LockSupport.parkNanos(this, TimeUnit.MILLISECONDS.toNanos(iterationSleepMillis));
            this.sleepDebtNanos = 0L;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private long flushAtIORatio(double ratio) {
        Thread thread = Thread.currentThread();
        long sleepPaymentThreshold = TimeUnit.MILLISECONDS.toNanos(backgroundFlushSleepDebtThreshold);
        boolean seenDirtyPages = false;
        boolean flushedPages = false;
        double sleepFactor = (1.0 - ratio) / ratio;
        try (MajorFlushEvent event = this.tracer.beginCacheFlush();){
            for (MuninnPage page : this.pages) {
                if (page == null || thread.isInterrupted()) {
                    thread.interrupt();
                    long l = 0L;
                    return l;
                }
                boolean thisPageIsDirty = false;
                if (page.isWriteLocked() || !(thisPageIsDirty = page.isDirty()) || !page.decrementUsage()) {
                    seenDirtyPages |= thisPageIsDirty;
                    continue;
                }
                long stamp = page.tryReadLock();
                if (stamp != 0L) {
                    try {
                        if (!page.isDirty()) continue;
                        long startNanos = System.nanoTime();
                        page.flush(event.flushEventOpportunity());
                        long elapsedNanos = System.nanoTime() - startNanos;
                        this.sleepDebtNanos = (long)((double)this.sleepDebtNanos + (double)elapsedNanos * sleepFactor);
                        flushedPages = true;
                    }
                    catch (Throwable throwable) {
                    }
                    finally {
                        page.unlockRead(stamp);
                    }
                }
                if (this.sleepDebtNanos > sleepPaymentThreshold) {
                    LockSupport.parkNanos(this.sleepDebtNanos);
                    this.sleepDebtNanos = 0L;
                }
                this.checkBackgroundFlushPause();
            }
        }
        return seenDirtyPages ? (flushedPages ? backgroundFlushMediumBreak : backgroundFlushBusyBreak) : backgroundFlushLongBreak;
    }

    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("MuninnPageCache[ \n");
        for (MuninnPage page : this.pages) {
            sb.append(' ').append(page).append('\n');
        }
        sb.append(']').append('\n');
        return sb.toString();
    }
}

