/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.commandline.dbms;

import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.StringJoiner;
import org.eclipse.collections.api.set.MutableSet;
import org.eclipse.collections.impl.set.mutable.MutableSetFactoryImpl;
import org.neo4j.cli.AbstractAdminCommand;
import org.neo4j.cli.CommandFailedException;
import org.neo4j.cli.Converters;
import org.neo4j.cli.ExecutionContext;
import org.neo4j.commandline.Util;
import org.neo4j.commandline.dbms.LoadDumpExecutor;
import org.neo4j.configuration.Config;
import org.neo4j.configuration.GraphDatabaseSettings;
import org.neo4j.configuration.helpers.DatabaseNamePattern;
import org.neo4j.dbms.archive.Loader;
import org.neo4j.function.ThrowingSupplier;
import org.neo4j.internal.helpers.Exceptions;
import org.neo4j.io.fs.FileSystemAbstraction;
import picocli.CommandLine;

@CommandLine.Command(name="load", header={"Load a database from an archive created with the dump command."}, description={"Load a database from an archive. <archive-path> must be a directory containing an archive(s) created with the dump command. If neither --from-path or --from-stdin is supplied `server.directories.dumps.root` setting will be searched for the archive. Existing databases can be replaced by specifying --overwrite-destination. It is not possible to replace a database that is mounted in a running Neo4j server. If --info is specified, then the database is not loaded, but information (i.e. file count, byte count, and format of load file) about the archive is printed instead."})
public class LoadCommand
extends AbstractAdminCommand {
    @CommandLine.Parameters(arity="1", description={"Name of the database to load. Can contain * and ? for globbing. Note that * and ? have special meaning in some shells and might need to be escaped or used with quotes."}, converter={Converters.DatabaseNamePatternConverter.class})
    private DatabaseNamePattern database;
    @CommandLine.ArgGroup
    private SourceOption source = new SourceOption();
    @CommandLine.Option(names={"--overwrite-destination"}, arity="0..1", paramLabel="true|false", fallbackValue="true", showDefaultValue=CommandLine.Help.Visibility.ALWAYS, description={"If an existing database should be replaced."})
    private boolean force;
    @CommandLine.Option(names={"--info"}, fallbackValue="true", description={"Print meta-data information about the archive file, instead of loading the contained database."})
    private boolean info;
    private final Loader loader;
    static final String SYSTEM_ERR_MESSAGE = String.format("WARNING! You are loading a dump of Neo4j's internal system database.%nThis system database dump may contain unwanted metadata for the DBMS it was taken from;%nLoading it should only be done after consulting the Neo4j Operations Manual.%n", new Object[0]);

    public LoadCommand(ExecutionContext ctx, Loader loader) {
        super(ctx);
        this.loader = Objects.requireNonNull(loader);
    }

    protected Optional<String> commandConfigName() {
        return Optional.of("database-load");
    }

    public void execute() {
        if (this.source.path != null && !this.ctx.fs().isDirectory(Path.of(this.source.path, new String[0]))) {
            throw new CommandFailedException(this.source.path + " is not an existing directory");
        }
        if (this.database.containsPattern() && this.source.stdIn) {
            throw new CommandFailedException("Globbing in database name can not be used in combination with standard input. Specify a directory as source or a single target database");
        }
        Config config = this.buildConfig();
        if (this.source.path == null && !this.source.stdIn) {
            Path defaultDumpsPath = (Path)config.get(GraphDatabaseSettings.database_dumps_root_path);
            if (!this.ctx.fs().isDirectory(defaultDumpsPath)) {
                throw new CommandFailedException("The root location for storing dumps ('" + GraphDatabaseSettings.database_dumps_root_path.name() + "'=" + defaultDumpsPath + ") doesn't contain any dumps yet. Specify another directory with --from-path.");
            }
            this.source.path = defaultDumpsPath.toString();
        }
        Set<DumpInfo> dbNames = this.getDbNames(this.ctx.fs());
        if (this.info) {
            this.inspectDump(dbNames);
        } else {
            try {
                this.loadDump(dbNames, config);
            }
            catch (IOException e) {
                Util.wrapIOException((IOException)e);
            }
        }
    }

    private void inspectDump(Set<DumpInfo> dbNames) {
        ArrayList<FailedLoad> failedLoads = new ArrayList<FailedLoad>();
        for (DumpInfo dbName : dbNames) {
            try {
                Loader.DumpMetaData metaData = this.loader.getMetaData(this.getArchiveInputStreamSupplier(dbName.dumpPath));
                this.ctx.out().println("Database: " + dbName.dbName);
                this.ctx.out().println("Format: " + metaData.format());
                this.ctx.out().println("Files: " + metaData.fileCount());
                this.ctx.out().println("Bytes: " + metaData.byteCount());
                this.ctx.out().println();
            }
            catch (Exception e) {
                this.ctx.err().printf("Failed to get metadata for dump '%s': %s", dbName.dumpPath, e.getMessage());
                failedLoads.add(new FailedLoad(dbName.dbName, e));
            }
        }
        this.checkFailure(failedLoads, "Print metadata failed for databases: '");
    }

    private ThrowingSupplier<InputStream, IOException> getArchiveInputStreamSupplier(Path path) {
        if (path != null) {
            return () -> Files.newInputStream(path, new OpenOption[0]);
        }
        return () -> ((ExecutionContext)this.ctx).in();
    }

    protected void loadDump(Set<DumpInfo> dbNames, Config config) throws IOException {
        LoadDumpExecutor loadDumpExecutor = new LoadDumpExecutor(config, this.ctx.fs(), this.ctx.err(), this.loader);
        ArrayList<FailedLoad> failedLoads = new ArrayList<FailedLoad>();
        for (DumpInfo dbName : dbNames) {
            try {
                if (dbName.dbName.equals("system")) {
                    this.ctx.err().print(SYSTEM_ERR_MESSAGE);
                }
                String dumpInputDescription = dbName.dumpPath == null ? "reading from stdin" : dbName.dumpPath.toString();
                ThrowingSupplier<InputStream, IOException> dumpInputStreamSupplier = this.getArchiveInputStreamSupplier(dbName.dumpPath);
                loadDumpExecutor.execute(new LoadDumpExecutor.DumpInput(dumpInputStreamSupplier, Optional.ofNullable(dbName.dumpPath), dumpInputDescription), dbName.dbName, this.force);
            }
            catch (Exception e) {
                this.ctx.err().printf("Failed to load database '%s': %s", dbName.dbName, e.getMessage());
                failedLoads.add(new FailedLoad(dbName.dbName, e));
            }
        }
        this.checkFailure(failedLoads, "Load failed for databases: '");
    }

    private void checkFailure(List<FailedLoad> failedLoads, String prefix) {
        if (!failedLoads.isEmpty()) {
            StringJoiner failedDbs = new StringJoiner("', '", prefix, "'");
            Exception exceptions = null;
            for (FailedLoad failedLoad : failedLoads) {
                failedDbs.add(failedLoad.dbName);
                exceptions = (Exception)Exceptions.chain(exceptions, (Throwable)failedLoad.e);
            }
            this.ctx.err().println(failedDbs);
            throw new CommandFailedException(failedDbs.toString(), exceptions);
        }
    }

    private Set<DumpInfo> getDbNames(FileSystemAbstraction fs) {
        if (this.source.stdIn) {
            return Set.of(new DumpInfo(this.database.getDatabaseName(), null));
        }
        Path dumpDir = Path.of(this.source.path, new String[0]);
        if (!this.database.containsPattern()) {
            return Set.of(new DumpInfo(this.database.getDatabaseName(), dumpDir.resolve(this.database.getDatabaseName() + ".dump")));
        }
        MutableSet dbNames = MutableSetFactoryImpl.INSTANCE.empty();
        try {
            for (Path path : fs.listFiles(dumpDir)) {
                String dbName;
                String fileName = path.getFileName().toString();
                if (fs.isDirectory(path) || !fileName.endsWith(".dump") || !this.database.matches(dbName = fileName.substring(0, fileName.length() - ".dump".length()))) continue;
                dbNames.add(new DumpInfo(dbName, path));
            }
        }
        catch (IOException e) {
            throw new CommandFailedException("Failed to list dump files", (Throwable)e);
        }
        if (dbNames.isEmpty()) {
            throw new CommandFailedException("Pattern '" + this.database.getDatabaseName() + "' did not match any dump file in " + dumpDir.toAbsolutePath());
        }
        return dbNames;
    }

    protected Config buildConfig() {
        return this.createPrefilledConfigBuilder().build();
    }

    private static class SourceOption {
        @CommandLine.Option(names={"--from-path"}, paramLabel="<path>", description={"Path to directory containing archive(s) created with the dump command."})
        private String path;
        @CommandLine.Option(names={"--from-stdin"}, description={"Read dump from standard input."})
        private boolean stdIn;

        private SourceOption() {
        }
    }

    protected record DumpInfo(String dbName, Path dumpPath) {
    }

    record FailedLoad(String dbName, Exception e) {
    }
}

