/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.dbms.diagnostics.profile;

import java.nio.file.Path;
import java.time.Duration;
import java.util.List;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.stream.Stream;
import org.neo4j.cli.AbstractAdminCommand;
import org.neo4j.cli.CommandFailedException;
import org.neo4j.cli.Converters;
import org.neo4j.cli.ExecutionContext;
import org.neo4j.configuration.Config;
import org.neo4j.configuration.SettingValueParsers;
import org.neo4j.dbms.archive.Dumper;
import org.neo4j.dbms.archive.StandardCompressionFormat;
import org.neo4j.dbms.diagnostics.jmx.JMXDumper;
import org.neo4j.dbms.diagnostics.jmx.JmxDump;
import org.neo4j.dbms.diagnostics.profile.JfrProfiler;
import org.neo4j.dbms.diagnostics.profile.JstackProfiler;
import org.neo4j.dbms.diagnostics.profile.ProfileTool;
import org.neo4j.dbms.diagnostics.profile.Profiler;
import org.neo4j.internal.helpers.collection.Iterators;
import org.neo4j.io.fs.FileHandle;
import org.neo4j.io.fs.FileSystemAbstraction;
import org.neo4j.kernel.diagnostics.NonInteractiveProgress;
import org.neo4j.time.Clocks;
import org.neo4j.time.Stopwatch;
import org.neo4j.time.SystemNanoClock;
import picocli.CommandLine;

@CommandLine.Command(name="profile", header={"Profile a running neo4j process"}, description={"Runs various profilers against a running neo4j VM. Note: This is a beta version. Behavior and surface will change in future versions."}, hidden=true)
public class ProfileCommand
extends AbstractAdminCommand {
    @CommandLine.Parameters(description={"Output directory of profiles"})
    private Path output;
    @CommandLine.Parameters(description={"Duration, how long the profilers should run"}, converter={Converters.DurationConverter.class})
    private Duration duration;
    @CommandLine.Parameters(description={"The selected profilers to run. Valid values: ${COMPLETION-CANDIDATES}"})
    private Set<ProfilerSource> profilers = Set.of(ProfilerSource.values());
    @CommandLine.Option(names={"--skip-compression"}, description={"Keeps the result in a directory structure instead of compressing"})
    private boolean skipCompression;

    public ProfileCommand(ExecutionContext ctx) {
        super(ctx);
    }

    protected void execute() throws Exception {
        if (this.duration.isNegative() || this.duration.isZero()) {
            this.ctx.out().println("Duration needs to be positive");
            return;
        }
        FileSystemAbstraction fs = this.ctx.fs();
        if (fs.isDirectory(this.output) && fs.listFiles(this.output).length > 0) {
            this.ctx.out().println("Output directory needs to be empty");
            return;
        }
        if (this.profilers.isEmpty()) {
            this.profilers = Set.of(ProfilerSource.values());
        }
        Config config = this.createPrefilledConfigBuilder().build();
        try (JmxDump jmxDump = this.getJmxDump(fs, config);){
            SystemNanoClock clock = Clocks.nanoClock();
            this.ctx.out().printf("Profilers %s selected. Duration %s. Output directory %s%n", this.profilers, SettingValueParsers.DURATION.valueToString((Object)this.duration), this.output.toAbsolutePath());
            try (ProfileTool tool = new ProfileTool();){
                if (this.profilers.contains((Object)ProfilerSource.JFR)) {
                    this.addProfiler(tool, new JfrProfiler(jmxDump, fs, this.output.resolve("jfr"), this.duration, clock));
                }
                if (this.profilers.contains((Object)ProfilerSource.THREADS)) {
                    this.addProfiler(tool, new JstackProfiler(jmxDump, fs, this.output.resolve("threads"), Duration.ofSeconds(1L), clock));
                }
                this.installShutdownHook(tool);
                tool.start();
                Stopwatch stopwatch = Stopwatch.start();
                NonInteractiveProgress progress = new NonInteractiveProgress(this.ctx.out(), true);
                while (!stopwatch.hasTimedOut(this.duration) && tool.hasRunningProfilers()) {
                    Thread.sleep(10L);
                    progress.percentChanged((int)(stopwatch.elapsed(TimeUnit.MILLISECONDS) * 100L / this.duration.toMillis()));
                }
                tool.stop();
                progress.finished();
                List profilers = Iterators.asList(tool.profilers().iterator());
                if (!stopwatch.hasTimedOut(this.duration)) {
                    this.ctx.out().println("Profiler stopped before expected duration.");
                }
                if (!profilers.isEmpty()) {
                    profilers.forEach(this::printFailedProfiler);
                    if (profilers.stream().allMatch(profiler -> profiler.failure() != null)) {
                        this.ctx.out().println("All profilers failed.");
                    } else {
                        this.ctx.out().println("Profiler results:");
                        for (Path path2 : fs.listFiles(this.output)) {
                            try (Stream fileStream = fs.streamFilesRecursive(path2, false);){
                                List<Path> files = fileStream.map(FileHandle::getRelativePath).toList();
                                this.ctx.out().printf("%s/ [%d %s]%n", this.output.relativize(path2), files.size(), files.size() > 1 ? "files" : "file");
                                int numFilesToPrint = 3;
                                for (int i = 0; i < files.size() && i <= numFilesToPrint; ++i) {
                                    if (i < numFilesToPrint) {
                                        this.ctx.out().printf("\t%s%n", files.get(i));
                                        continue;
                                    }
                                    this.ctx.out().printf("\t...%n", new Object[0]);
                                }
                            }
                        }
                        if (!this.skipCompression) {
                            Path archive = this.output.resolve(String.format("profile-%s.gzip", clock.instant()));
                            this.ctx.out().printf("%nCompressing result into %s", archive.getFileName());
                            Dumper dumper = new Dumper(this.ctx.out());
                            dumper.dump(this.output, this.output, dumper.openForDump(archive), StandardCompressionFormat.GZIP, path -> path.equals(archive));
                            for (Path path3 : fs.listFiles(this.output, arg_0 -> ((FileSystemAbstraction)fs).isDirectory(arg_0))) {
                                fs.deleteRecursively(path3);
                            }
                        }
                    }
                }
            }
        }
    }

    private JmxDump getJmxDump(FileSystemAbstraction fs, Config config) {
        return new JMXDumper(config, fs, this.ctx.out(), this.ctx.err(), this.verbose).getJMXDump().orElseThrow(() -> new CommandFailedException("Can not connect to running Neo4j. Profiling can not be done"));
    }

    private void installShutdownHook(ProfileTool tool) {
        Runtime.getRuntime().addShutdownHook(new Thread(tool::stop));
    }

    private void addProfiler(ProfileTool tool, Profiler profiler) {
        if (!tool.add(profiler)) {
            this.ctx.out().println(profiler.getClass().getSimpleName() + " is not available and will not be used");
        }
    }

    private void printFailedProfiler(Profiler profiler) {
        if (profiler.failure() != null) {
            this.ctx.out().println(profiler.getClass().getSimpleName() + " failed with: " + profiler.failure().getMessage());
            if (this.verbose) {
                profiler.failure().printStackTrace(this.ctx.err());
            }
        }
    }

    public static enum ProfilerSource {
        JFR,
        THREADS;

    }
}

