/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.commandline.dbms;

import java.io.IOException;
import java.nio.file.DirectoryStream;
import java.nio.file.Path;
import java.util.Collection;
import java.util.Locale;
import java.util.function.Predicate;
import java.util.function.ToDoubleFunction;
import org.neo4j.cli.AbstractCommand;
import org.neo4j.cli.CommandFailedException;
import org.neo4j.cli.Converters;
import org.neo4j.cli.ExecutionContext;
import org.neo4j.configuration.Config;
import org.neo4j.configuration.ConfigUtils;
import org.neo4j.configuration.ExternalSettings;
import org.neo4j.configuration.GraphDatabaseInternalSettings;
import org.neo4j.configuration.GraphDatabaseSettings;
import org.neo4j.graphdb.config.Setting;
import org.neo4j.io.ByteUnit;
import org.neo4j.io.fs.FileSystemAbstraction;
import org.neo4j.io.layout.DatabaseLayout;
import org.neo4j.io.layout.Neo4jLayout;
import org.neo4j.io.os.OsBeanUtil;
import org.neo4j.kernel.api.index.IndexDirectoryStructure;
import org.neo4j.kernel.internal.NativeIndexFileFilter;
import org.neo4j.storageengine.api.StorageEngineFactory;
import picocli.CommandLine;

@CommandLine.Command(name="memrec", header={"Print Neo4j heap and pagecache memory settings recommendations."}, description={"Print heuristic memory setting recommendations for the Neo4j JVM heap and pagecache. The heuristic is based on the total memory of the system the command is running on, or on the amount of memory specified with the --memory argument. The heuristic assumes that the system is dedicated to running Neo4j. If this is not the case, then use the --memory argument to specify how much memory can be expected to be dedicated to Neo4j. The output is formatted such that it can be copy-pasted into the neo4j.conf file."})
class MemoryRecommendationsCommand
extends AbstractCommand {
    private static final Bracket[] datapoints = new Bracket[]{new Bracket(0.01, 0.007, 0.002), new Bracket(1.0, 0.65, 0.3), new Bracket(2.0, 1.0, 0.5), new Bracket(4.0, 1.5, 2.0), new Bracket(6.0, 2.0, 3.0), new Bracket(8.0, 2.5, 3.5), new Bracket(10.0, 3.0, 4.0), new Bracket(12.0, 3.5, 4.5), new Bracket(16.0, 4.0, 5.0), new Bracket(24.0, 6.0, 8.0), new Bracket(32.0, 8.0, 12.0), new Bracket(64.0, 12.0, 24.0), new Bracket(128.0, 16.0, 31.0), new Bracket(256.0, 20.0, 31.0), new Bracket(512.0, 24.0, 31.0), new Bracket(1024.0, 30.0, 31.0)};
    @CommandLine.Option(names={"--memory"}, paramLabel="<size>", converter={Converters.ByteUnitConverter.class}, description={"Recommend memory settings with respect to the given amount of memory, instead of the total memory of the system running the command."})
    private Long memory;
    @CommandLine.Option(names={"--docker"}, arity="0", description={"The recommended memory settings are produced in the form of environment variables that can be directly passed to Neo4j docker container."})
    private boolean dockerOutput;

    MemoryRecommendationsCommand(ExecutionContext ctx) {
        super(ctx);
    }

    static long recommendOsMemory(long totalMemoryBytes) {
        Brackets brackets = MemoryRecommendationsCommand.findMemoryBrackets(totalMemoryBytes);
        return brackets.recommend(Bracket::osMemory);
    }

    static long recommendHeapMemory(long totalMemoryBytes) {
        Brackets brackets = MemoryRecommendationsCommand.findMemoryBrackets(totalMemoryBytes);
        return brackets.recommend(Bracket::heapMemory);
    }

    static long recommendTxStateMemory(Config config, long heapMemoryBytes) {
        switch ((GraphDatabaseSettings.TransactionStateMemoryAllocation)config.get(GraphDatabaseSettings.tx_state_memory_allocation)) {
            case OFF_HEAP: {
                long recommendation = heapMemoryBytes / 4L;
                recommendation = Math.max(ByteUnit.mebiBytes((long)128L), recommendation);
                recommendation = Math.min(ByteUnit.gibiBytes((long)8L), recommendation);
                return recommendation;
            }
            case ON_HEAP: {
                return 0L;
            }
        }
        throw new IllegalArgumentException("Unsupported type of memory allocation.");
    }

    static long recommendPageCacheMemory(long totalMemoryBytes, long offHeapMemory) {
        long osMemory = MemoryRecommendationsCommand.recommendOsMemory(totalMemoryBytes);
        long heapMemory = MemoryRecommendationsCommand.recommendHeapMemory(totalMemoryBytes);
        long recommendation = totalMemoryBytes - osMemory - heapMemory - offHeapMemory;
        recommendation = Math.max(ByteUnit.mebiBytes((long)8L), recommendation);
        recommendation = Math.min(ByteUnit.tebiBytes((long)16L), recommendation);
        return recommendation;
    }

    private static Brackets findMemoryBrackets(long totalMemoryBytes) {
        double totalMemoryGB = (double)totalMemoryBytes / (double)ByteUnit.gibiBytes((long)1L);
        Bracket lower = null;
        Bracket upper = null;
        for (int i = 1; i < datapoints.length; ++i) {
            if (!(totalMemoryGB < MemoryRecommendationsCommand.datapoints[i].totalMemory)) continue;
            lower = datapoints[i - 1];
            upper = datapoints[i];
            break;
        }
        if (lower == null) {
            lower = datapoints[datapoints.length - 1];
            upper = datapoints[datapoints.length - 1];
        }
        return new Brackets(totalMemoryGB, lower, upper);
    }

    static String bytesToString(double bytes) {
        double gibi1 = ByteUnit.ONE_GIBI_BYTE;
        double mebi1 = ByteUnit.ONE_MEBI_BYTE;
        double mebi100 = 100.0 * mebi1;
        double kibi1 = ByteUnit.ONE_KIBI_BYTE;
        double kibi100 = 100.0 * kibi1;
        if (bytes >= gibi1) {
            double gibibytes = bytes / gibi1;
            double modMebi = bytes % gibi1;
            if (modMebi >= mebi100) {
                return String.format(Locale.ROOT, "%dm", Math.round(bytes / mebi100) * 100L);
            }
            return String.format(Locale.ROOT, "%.0fg", gibibytes);
        }
        if (bytes >= mebi1) {
            double mebibytes = bytes / mebi1;
            double modKibi = bytes % mebi1;
            if (modKibi >= kibi100) {
                return String.format(Locale.ROOT, "%dk", Math.round(bytes / kibi100) * 100L);
            }
            return String.format(Locale.ROOT, "%.0fm", mebibytes);
        }
        double kibiBytes = bytes / kibi1;
        return String.format(Locale.ROOT, "%dk", (long)Math.ceil(kibiBytes));
    }

    protected void execute() {
        if (this.memory == null) {
            this.memory = OsBeanUtil.getTotalPhysicalMemory();
        }
        Path configFile = this.ctx.confDir().resolve("neo4j.conf");
        Config config = this.getConfig(configFile);
        long offHeapMemory = MemoryRecommendationsCommand.recommendTxStateMemory(config, this.memory);
        String os = MemoryRecommendationsCommand.bytesToString(MemoryRecommendationsCommand.recommendOsMemory(this.memory));
        String heap = MemoryRecommendationsCommand.bytesToString(MemoryRecommendationsCommand.recommendHeapMemory(this.memory));
        String pagecache = MemoryRecommendationsCommand.bytesToString(MemoryRecommendationsCommand.recommendPageCacheMemory(this.memory, offHeapMemory));
        String txState = MemoryRecommendationsCommand.bytesToString(offHeapMemory);
        Path databasesRoot = (Path)config.get(GraphDatabaseInternalSettings.databases_root_path);
        Neo4jLayout storeLayout = Neo4jLayout.of((Config)config);
        Collection layouts = storeLayout.databaseLayouts();
        long pageCacheSize = this.pageCacheSize(layouts);
        long luceneSize = this.luceneSize(layouts);
        this.print("# Memory settings recommendation from neo4j-admin memrec:");
        this.print("#");
        this.print("# Assuming the system is dedicated to running Neo4j and has " + ByteUnit.bytesToString((long)this.memory) + " of memory,");
        this.print("# we recommend a heap size of around " + heap + ", and a page cache of around " + pagecache + ",");
        this.print("# and that about " + os + " is left for the operating system, and the native memory");
        this.print("# needed by Lucene and Netty.");
        this.print("#");
        this.print("# Tip: If the indexing storage use is high, e.g. there are many indexes or most");
        this.print("# data indexed, then it might advantageous to leave more memory for the");
        this.print("# operating system.");
        this.print("#");
        this.print("# Tip: Depending on the workload type you may want to increase the amount");
        this.print("# of off-heap memory available for storing transaction state.");
        this.print("# For instance, in case of large write-intensive transactions");
        this.print("# increasing it can lower GC overhead and thus improve performance.");
        this.print("# On the other hand, if vast majority of transactions are small or read-only");
        this.print("# then you can decrease it and increase page cache instead.");
        this.print("#");
        this.print("# Tip: The more concurrent transactions your workload has and the more updates");
        this.print("# they do, the more heap memory you will need. However, don't allocate more");
        this.print("# than 31g of heap, since this will disable pointer compression, also known as");
        this.print("# \"compressed oops\", in the JVM and make less effective use of the heap.");
        this.print("#");
        this.print("# Tip: Setting the initial and the max heap size to the same value means the");
        this.print("# JVM will never need to change the heap size. Changing the heap size otherwise");
        this.print("# involves a full GC, which is desirable to avoid.");
        this.print("#");
        this.print("# Based on the above, the following memory settings are recommended:");
        this.print(ExternalSettings.initial_heap_size, heap);
        this.print(ExternalSettings.max_heap_size, heap);
        this.print(GraphDatabaseSettings.pagecache_memory, pagecache);
        if (offHeapMemory != 0L) {
            this.print(GraphDatabaseSettings.tx_state_max_off_heap_memory, txState);
        }
        this.print("#");
        this.print("# It is also recommended turning out-of-memory errors into full crashes,");
        this.print("# instead of allowing a partially crashed database to continue running:");
        this.print(ExternalSettings.additional_jvm + "-XX:+ExitOnOutOfMemoryError");
        this.print("#");
        this.print("# The numbers below have been derived based on your current databases located at: '" + databasesRoot + "'.");
        this.print("# They can be used as an input into more detailed memory analysis.");
        this.print("# Total size of lucene indexes in all databases: " + MemoryRecommendationsCommand.bytesToString(luceneSize));
        this.print("# Total size of data and native indexes in all databases: " + MemoryRecommendationsCommand.bytesToString(pageCacheSize));
    }

    private void print(Setting<?> setting, String value) {
        if (!this.dockerOutput) {
            this.print(setting.name() + "=" + value);
        } else {
            String nameWithFixedUnderscores = setting.name().replaceAll("_", "__");
            String nameWithFixedUnderscoresAndDots = nameWithFixedUnderscores.replaceAll("\\.", "_");
            this.print("EXPORT NEO4J_" + nameWithFixedUnderscoresAndDots + "='" + value + "'");
        }
    }

    private long pageCacheSize(Collection<DatabaseLayout> layouts) {
        return layouts.stream().mapToLong(this::getDatabasePageCacheSize).sum();
    }

    private long getDatabasePageCacheSize(DatabaseLayout layout) {
        return this.sumStoreFiles(layout) + this.sumIndexFiles(IndexDirectoryStructure.baseSchemaIndexFolder((Path)layout.databaseDirectory()), this.getNativeIndexFileFilter(layout.databaseDirectory(), false));
    }

    private long luceneSize(Collection<DatabaseLayout> layouts) {
        return layouts.stream().mapToLong(this::getDatabaseLuceneSize).sum();
    }

    private long getDatabaseLuceneSize(DatabaseLayout databaseLayout) {
        Path databaseDirectory = databaseLayout.databaseDirectory();
        return this.sumIndexFiles(IndexDirectoryStructure.baseSchemaIndexFolder((Path)databaseDirectory), this.getNativeIndexFileFilter(databaseDirectory, true));
    }

    private DirectoryStream.Filter<Path> getNativeIndexFileFilter(Path storeDir, boolean inverse) {
        NativeIndexFileFilter nativeIndexFilter = new NativeIndexFileFilter(storeDir);
        return arg_0 -> this.lambda$getNativeIndexFileFilter$0(inverse, (Predicate)nativeIndexFilter, arg_0);
    }

    private long sumStoreFiles(DatabaseLayout databaseLayout) {
        StorageEngineFactory storageEngineFactory = StorageEngineFactory.selectStorageEngine();
        FileSystemAbstraction fileSystem = this.ctx.fs();
        try {
            long total = storageEngineFactory.listStorageFiles(fileSystem, databaseLayout).stream().mapToLong(arg_0 -> ((FileSystemAbstraction)fileSystem).getFileSize(arg_0)).sum();
            return total += this.sizeOfFileIfExists(databaseLayout.labelScanStore());
        }
        catch (IOException e) {
            return 0L;
        }
    }

    private long sizeOfFileIfExists(Path file) {
        FileSystemAbstraction fileSystem = this.ctx.fs();
        return fileSystem.fileExists(file) ? fileSystem.getFileSize(file) : 0L;
    }

    private long sumIndexFiles(Path file, DirectoryStream.Filter<Path> filter) {
        long total = 0L;
        if (this.ctx.fs().isDirectory(file)) {
            Path[] children = this.ctx.fs().listFiles(file, filter);
            if (children != null) {
                for (Path child : children) {
                    total += this.sumIndexFiles(child, filter);
                }
            }
        } else {
            total += this.ctx.fs().getFileSize(file);
        }
        return total;
    }

    private Config getConfig(Path configFile) {
        if (!this.ctx.fs().fileExists(configFile)) {
            throw new CommandFailedException("Unable to find config file, tried: " + configFile.toAbsolutePath());
        }
        try {
            Config config = Config.newBuilder().fromFile(configFile).set(GraphDatabaseSettings.neo4j_home, (Object)this.ctx.homeDir().toAbsolutePath()).build();
            ConfigUtils.disableAllConnectors((Config)config);
            return config;
        }
        catch (Exception e) {
            throw new CommandFailedException("Failed to read config file: " + configFile.toAbsolutePath(), (Throwable)e);
        }
    }

    private void print(String text) {
        this.ctx.out().println(text);
    }

    private /* synthetic */ boolean lambda$getNativeIndexFileFilter$0(boolean inverse, Predicate nativeIndexFilter, Path file) throws IOException {
        if (this.ctx.fs().isDirectory(file)) {
            return true;
        }
        if (file.getFileName().toString().equals("failure-message")) {
            return false;
        }
        return inverse != nativeIndexFilter.test(file);
    }

    private static final class Brackets {
        private final double totalMemoryGB;
        private final Bracket lower;
        private final Bracket upper;

        private Brackets(double totalMemoryGB, Bracket lower, Bracket upper) {
            this.totalMemoryGB = totalMemoryGB;
            this.lower = lower;
            this.upper = upper;
        }

        private double differenceFactor() {
            if (this.lower == this.upper) {
                return 0.0;
            }
            return (this.totalMemoryGB - this.lower.totalMemory) / (this.upper.totalMemory - this.lower.totalMemory);
        }

        public long recommend(ToDoubleFunction<Bracket> parameter) {
            double factor = this.differenceFactor();
            double lowerParam = parameter.applyAsDouble(this.lower);
            double upperParam = parameter.applyAsDouble(this.upper);
            double diff = upperParam - lowerParam;
            double recommend = lowerParam + diff * factor;
            return ByteUnit.mebiBytes((long)((long)(recommend * 1024.0)));
        }
    }

    private static final class Bracket {
        private final double totalMemory;
        private final double osMemory;
        private final double heapMemory;

        private Bracket(double totalMemory, double osMemory, double heapMemory) {
            this.totalMemory = totalMemory;
            this.osMemory = osMemory;
            this.heapMemory = heapMemory;
        }

        double osMemory() {
            return this.osMemory;
        }

        double heapMemory() {
            return this.heapMemory;
        }
    }
}

