/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.csv.reader;

import java.io.BufferedInputStream;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.PushbackInputStream;
import java.io.Reader;
import java.io.StringReader;
import java.io.UncheckedIOException;
import java.nio.charset.Charset;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Enumeration;
import java.util.function.LongSupplier;
import java.util.zip.GZIPInputStream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import java.util.zip.ZipInputStream;
import org.neo4j.cloud.storage.StoragePath;
import org.neo4j.cloud.storage.StorageSettingsDeclaration;
import org.neo4j.cloud.storage.StorageUtils;
import org.neo4j.collection.RawIterator;
import org.neo4j.csv.reader.BufferedCharSeeker;
import org.neo4j.csv.reader.CharReadable;
import org.neo4j.csv.reader.Configuration;
import org.neo4j.csv.reader.Magic;
import org.neo4j.csv.reader.MagicInputStream;
import org.neo4j.csv.reader.MultiReadable;
import org.neo4j.csv.reader.WrappedCharReadable;
import org.neo4j.function.IOFunction;
import org.neo4j.function.ThrowingFunction;

public class Readables {
    private Readables() {
        throw new AssertionError((Object)"No instances allowed");
    }

    public static CharReadable wrap(InputStream stream, String sourceName, Charset charset) throws IOException {
        return Readables.wrap(stream, sourceName, charset, 0L);
    }

    public static CharReadable wrap(InputStream stream, String sourceName, Charset charset, long length) throws IOException {
        byte[] bytes = new byte[Magic.longest()];
        PushbackInputStream pushbackStream = new PushbackInputStream(stream, bytes.length);
        Charset usedCharset = charset;
        int read = stream.read(bytes);
        if (read >= 0) {
            bytes = read < bytes.length ? Arrays.copyOf(bytes, read) : bytes;
            Magic magic = Magic.of(bytes);
            int excessiveBytes = read;
            if (magic.impliesEncoding()) {
                excessiveBytes -= magic.length();
                usedCharset = magic.encoding();
            }
            pushbackStream.unread(bytes, read - excessiveBytes, excessiveBytes);
            if (magic == Magic.ZIP) {
                return Readables.zipReadable(pushbackStream, usedCharset, sourceName);
            }
            if (magic == Magic.GZIP) {
                return Readables.gzipReadable(pushbackStream, usedCharset, sourceName, length);
            }
        }
        return Readables.readable(pushbackStream, usedCharset, sourceName, length);
    }

    public static CharReadable wrap(String sourceDescription, String data) {
        return Readables.wrap(sourceDescription, new StringReader(data), data.length());
    }

    public static CharReadable wrap(String data) {
        return Readables.wrap(new StringReader(data), data.length());
    }

    public static CharReadable wrap(Reader reader, long length) {
        return Readables.wrap(reader.toString(), reader, length);
    }

    public static CharReadable wrap(String sourceDescription, Reader reader, long length) {
        return new WrappedCharReadable(length, reader, sourceDescription);
    }

    private static CharReadable zipReadable(InputStream input, Charset charset, String sourceName) throws IOException {
        ZipEntry entry;
        ZipInputStream stream = new ZipInputStream(input);
        while ((entry = stream.getNextEntry()) != null) {
            if (entry.isDirectory() || Readables.invalidZipEntry(entry.getName())) continue;
            return Readables.readable(stream, charset, sourceName, entry.getSize());
        }
        stream.close();
        throw new IllegalStateException("Couldn't find zip entry when opening the stream at " + sourceName);
    }

    private static CharReadable zipReadableFromFile(Path path, Charset charset, String sourceName) throws IOException {
        try (ZipFile zipFile = new ZipFile(path.toFile());){
            ZipEntry entry = Readables.getSingleSuitableEntry(zipFile);
            CharReadable charReadable = Readables.readable(Readables.openZipInputStream(path, entry), charset, sourceName, entry.getSize());
            return charReadable;
        }
    }

    private static CharReadable gzipReadable(InputStream input, Charset charset, String sourceName, long fileSize) throws IOException {
        final LongSupplier[] bytesReadFromCompressedSource = new LongSupplier[1];
        GZIPInputStream zipStream = new GZIPInputStream(input){
            {
                super(in);
                bytesReadFromCompressedSource[0] = this.inf::getBytesRead;
            }
        };
        return new WrappedCharReadable(fileSize, Readables.reader(zipStream, charset, sourceName), sourceName){

            @Override
            public float compressionRatio() {
                long decompressedPosition = this.position();
                long compressedPosition = bytesReadFromCompressedSource[0].getAsLong();
                return (float)((double)compressedPosition / (double)decompressedPosition);
            }
        };
    }

    private static CharReadable readableWithEncoding(MagicInputStream input, Charset defaultCharset, String sourceName) throws IOException {
        Magic magic = input.magic();
        Charset usedCharset = defaultCharset;
        if (magic.impliesEncoding()) {
            long skip = input.skip(magic.length());
            if (skip != (long)magic.length()) {
                throw new IOException("Unable to skip " + magic.length() + " bytes, only able to skip " + skip + " bytes.");
            }
            usedCharset = magic.encoding();
        }
        return Readables.readable(input, usedCharset, sourceName, Files.size(input.path().toAbsolutePath()));
    }

    private static Reader reader(InputStream input, Charset charset, final String sourceName) {
        return new InputStreamReader(input, charset){

            public String toString() {
                return sourceName;
            }
        };
    }

    private static CharReadable readable(InputStream input, Charset charset, String sourceName, long size) {
        return Readables.wrap(Readables.reader(input, charset, sourceName), size);
    }

    private static ZipInputStream openZipInputStream(Path path, ZipEntry entry) throws IOException {
        ZipEntry readEntry;
        ZipInputStream stream = new ZipInputStream(new BufferedInputStream(new FileInputStream(path.toFile())));
        while ((readEntry = stream.getNextEntry()) != null) {
            if (readEntry.isDirectory() || !readEntry.getName().equals(entry.getName())) continue;
            return stream;
        }
        stream.close();
        throw new IllegalStateException("Couldn't find zip entry with name " + entry.getName() + " when opening it as a stream");
    }

    private static ZipEntry getSingleSuitableEntry(ZipFile zipFile) throws IOException {
        ArrayList<String> unsuitableEntries = new ArrayList<String>();
        Enumeration<? extends ZipEntry> enumeration = zipFile.entries();
        ZipEntry found = null;
        while (enumeration.hasMoreElements()) {
            ZipEntry entry = enumeration.nextElement();
            if (entry.isDirectory() || Readables.invalidZipEntry(entry.getName())) {
                unsuitableEntries.add(entry.getName());
                continue;
            }
            if (found != null) {
                throw new IOException("Multiple suitable files found in zip file " + zipFile.getName() + ", at least " + found.getName() + " and " + entry.getName() + ". Only a single file per zip file is supported");
            }
            found = entry;
        }
        if (found == null) {
            throw new IOException("No suitable file found in zip file " + zipFile.getName() + "." + (String)(!unsuitableEntries.isEmpty() ? " Although found these unsuitable entries " + String.valueOf(unsuitableEntries) : ""));
        }
        return found;
    }

    private static boolean invalidZipEntry(String name) {
        return name.contains("__MACOSX") || name.startsWith(".") || name.contains("/.");
    }

    public static RawIterator<CharReadable, IOException> individualFiles(Configuration config, Charset charset, Path ... files) {
        return Readables.iterator(new FromFile(charset, config.readIsForSampling()), files);
    }

    public static CharReadable files(Charset charset, Path ... files) throws IOException {
        FromFile opener = new FromFile(charset, false);
        return switch (files.length) {
            case 0 -> CharReadable.EMPTY;
            case 1 -> (CharReadable)opener.apply(files[0]);
            default -> new MultiReadable(Readables.iterator(opener, files));
        };
    }

    @SafeVarargs
    public static <IN, OUT> RawIterator<OUT, IOException> iterator(final ThrowingFunction<IN, OUT, IOException> converter, final IN ... items) {
        if (items.length == 0) {
            throw new IllegalStateException("No source items specified");
        }
        return new RawIterator<OUT, IOException>(){
            private int cursor;

            public boolean hasNext() {
                return this.cursor < items.length;
            }

            public OUT next() throws IOException {
                if (!this.hasNext()) {
                    throw new IllegalStateException();
                }
                return converter.apply(items[this.cursor++]);
            }
        };
    }

    public static char[] extractFirstLineFrom(CharReadable source) throws IOException {
        return Readables.extractFirstLineFrom((char[] into, int offset) -> source.read(into, offset, 1) > 0);
    }

    public static char[] extractFirstLineFrom(char[] data, int offset, int length) {
        try {
            return Readables.extractFirstLineFrom((char[] into, int intoOffset) -> {
                if (intoOffset < length) {
                    into[intoOffset] = data[offset + intoOffset];
                    return true;
                }
                return false;
            });
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    public static char[] extractFirstLineFrom(CharSupplier source) throws IOException {
        boolean foundEol;
        char[] result = new char[100];
        int cursor = 0;
        do {
            if (cursor >= result.length) {
                result = Arrays.copyOf(result, cursor * 2);
            }
            if (!source.next(result, cursor)) break;
            foundEol = BufferedCharSeeker.isEolChar(result[cursor]);
            if (foundEol) continue;
            ++cursor;
        } while (!foundEol);
        return Arrays.copyOf(result, cursor);
    }

    private record FromFile(Charset charset, boolean readIsForSampling) implements IOFunction<Path, CharReadable>
    {
        public CharReadable apply(Path path) throws IOException {
            MagicInputStream input = MagicInputStream.create(this.adaptPath(path));
            String sourceName = StorageUtils.toString((Path)path.toAbsolutePath());
            if (input.magic() == Magic.ZIP) {
                return input.isDefaultFileSystemBased() ? Readables.zipReadableFromFile(input.path(), this.charset, sourceName) : Readables.zipReadable(input, this.charset, sourceName);
            }
            if (input.magic() == Magic.GZIP) {
                return Readables.gzipReadable(input, this.charset, sourceName, Files.size(path));
            }
            return Readables.readableWithEncoding(input, this.charset, sourceName);
        }

        private Path adaptPath(Path path) {
            Path path2;
            if (this.readIsForSampling && path instanceof StoragePath) {
                StoragePath sp = (StoragePath)path;
                path2 = StorageSettingsDeclaration.adaptPathForSampling((StoragePath)sp);
            } else {
                path2 = path;
            }
            return path2;
        }
    }

    public static interface CharSupplier {
        public boolean next(char[] var1, int var2) throws IOException;
    }
}

