/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.driver.internal.async.pool;

import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicBoolean;
import org.neo4j.driver.Logger;
import org.neo4j.driver.Logging;
import org.neo4j.driver.exceptions.ClientException;
import org.neo4j.driver.exceptions.ServiceUnavailableException;
import org.neo4j.driver.internal.BoltServerAddress;
import org.neo4j.driver.internal.async.connection.ChannelAttributes;
import org.neo4j.driver.internal.async.connection.ChannelConnector;
import org.neo4j.driver.internal.async.pool.ConnectionFactory;
import org.neo4j.driver.internal.async.pool.ExtendedChannelPool;
import org.neo4j.driver.internal.async.pool.NettyChannelHealthChecker;
import org.neo4j.driver.internal.async.pool.NettyChannelPool;
import org.neo4j.driver.internal.async.pool.NettyChannelTracker;
import org.neo4j.driver.internal.async.pool.NetworkConnectionFactory;
import org.neo4j.driver.internal.async.pool.PoolSettings;
import org.neo4j.driver.internal.metrics.ListenerEvent;
import org.neo4j.driver.internal.metrics.MetricsListener;
import org.neo4j.driver.internal.shaded.io.netty.bootstrap.Bootstrap;
import org.neo4j.driver.internal.shaded.io.netty.channel.Channel;
import org.neo4j.driver.internal.shaded.io.netty.channel.EventLoopGroup;
import org.neo4j.driver.internal.spi.Connection;
import org.neo4j.driver.internal.spi.ConnectionPool;
import org.neo4j.driver.internal.util.Clock;
import org.neo4j.driver.internal.util.Futures;

public class ConnectionPoolImpl
implements ConnectionPool {
    private final ChannelConnector connector;
    private final Bootstrap bootstrap;
    private final NettyChannelTracker nettyChannelTracker;
    private final NettyChannelHealthChecker channelHealthChecker;
    private final PoolSettings settings;
    private final Logger log;
    private final MetricsListener metricsListener;
    private final boolean ownsEventLoopGroup;
    private final ConcurrentMap<BoltServerAddress, ExtendedChannelPool> pools = new ConcurrentHashMap<BoltServerAddress, ExtendedChannelPool>();
    private final AtomicBoolean closed = new AtomicBoolean();
    private final CompletableFuture<Void> closeFuture = new CompletableFuture();
    private final ConnectionFactory connectionFactory;

    public ConnectionPoolImpl(ChannelConnector connector, Bootstrap bootstrap, PoolSettings settings, MetricsListener metricsListener, Logging logging, Clock clock, boolean ownsEventLoopGroup) {
        this(connector, bootstrap, new NettyChannelTracker(metricsListener, bootstrap.config().group().next(), logging), new NettyChannelHealthChecker(settings, clock, logging), settings, metricsListener, logging, clock, ownsEventLoopGroup, new NetworkConnectionFactory(clock, metricsListener));
    }

    protected ConnectionPoolImpl(ChannelConnector connector, Bootstrap bootstrap, NettyChannelTracker nettyChannelTracker, NettyChannelHealthChecker nettyChannelHealthChecker, PoolSettings settings, MetricsListener metricsListener, Logging logging, Clock clock, boolean ownsEventLoopGroup, ConnectionFactory connectionFactory) {
        this.connector = connector;
        this.bootstrap = bootstrap;
        this.nettyChannelTracker = nettyChannelTracker;
        this.channelHealthChecker = nettyChannelHealthChecker;
        this.settings = settings;
        this.metricsListener = metricsListener;
        this.log = logging.getLog(ConnectionPool.class.getSimpleName());
        this.ownsEventLoopGroup = ownsEventLoopGroup;
        this.connectionFactory = connectionFactory;
    }

    @Override
    public CompletionStage<Connection> acquire(BoltServerAddress address) {
        this.log.trace("Acquiring a connection from pool towards %s", address);
        this.assertNotClosed();
        ExtendedChannelPool pool = this.getOrCreatePool(address);
        ListenerEvent acquireEvent = this.metricsListener.createListenerEvent();
        this.metricsListener.beforeAcquiringOrCreating(pool.id(), acquireEvent);
        CompletionStage<Channel> channelFuture = pool.acquire();
        return channelFuture.handle((channel, error) -> {
            try {
                this.processAcquisitionError(pool, address, (Throwable)error);
                this.assertNotClosed(address, (Channel)channel, pool);
                ChannelAttributes.setAuthorizationStateListener(channel, this.channelHealthChecker);
                Connection connection = this.connectionFactory.createConnection((Channel)channel, pool);
                this.metricsListener.afterAcquiredOrCreated(pool.id(), acquireEvent);
                Connection connection2 = connection;
                return connection2;
            }
            finally {
                this.metricsListener.afterAcquiringOrCreating(pool.id());
            }
        });
    }

    @Override
    public void retainAll(Set<BoltServerAddress> addressesToRetain) {
        for (BoltServerAddress address : this.pools.keySet()) {
            ExtendedChannelPool pool;
            int activeChannels;
            if (addressesToRetain.contains(address) || (activeChannels = this.nettyChannelTracker.inUseChannelCount(address)) != 0 || (pool = (ExtendedChannelPool)this.pools.remove(address)) == null) continue;
            this.log.info("Closing connection pool towards %s, it has no active connections and is not in the routing table registry.", address);
            this.closePoolInBackground(address, pool);
        }
    }

    @Override
    public int inUseConnections(BoltServerAddress address) {
        return this.nettyChannelTracker.inUseChannelCount(address);
    }

    @Override
    public int idleConnections(BoltServerAddress address) {
        return this.nettyChannelTracker.idleChannelCount(address);
    }

    @Override
    public CompletionStage<Void> close() {
        if (this.closed.compareAndSet(false, true)) {
            this.nettyChannelTracker.prepareToCloseChannels();
            CompletableFuture<Void> allPoolClosedFuture = this.closeAllPools();
            allPoolClosedFuture.whenComplete((ignored, pollCloseError) -> {
                this.pools.clear();
                if (!this.ownsEventLoopGroup) {
                    Futures.completeWithNullIfNoError(this.closeFuture, pollCloseError);
                } else {
                    this.shutdownEventLoopGroup((Throwable)pollCloseError);
                }
            });
        }
        return this.closeFuture;
    }

    @Override
    public boolean isOpen(BoltServerAddress address) {
        return this.pools.containsKey(address);
    }

    public String toString() {
        return "ConnectionPoolImpl{pools=" + this.pools + '}';
    }

    private void processAcquisitionError(ExtendedChannelPool pool, BoltServerAddress serverAddress, Throwable error) {
        Throwable cause = Futures.completionExceptionCause(error);
        if (cause != null) {
            if (cause instanceof TimeoutException) {
                this.metricsListener.afterTimedOutToAcquireOrCreate(pool.id());
                throw new ClientException("Unable to acquire connection from the pool within configured maximum time of " + this.settings.connectionAcquisitionTimeout() + "ms");
            }
            if (pool.isClosed()) {
                throw new ServiceUnavailableException(String.format("Connection pool for server %s is closed while acquiring a connection.", serverAddress), cause);
            }
            throw new CompletionException(cause);
        }
    }

    private void assertNotClosed() {
        if (this.closed.get()) {
            throw new IllegalStateException("Pool closed");
        }
    }

    private void assertNotClosed(BoltServerAddress address, Channel channel, ExtendedChannelPool pool) {
        if (this.closed.get()) {
            pool.release(channel);
            this.closePoolInBackground(address, pool);
            this.pools.remove(address);
            this.assertNotClosed();
        }
    }

    ExtendedChannelPool getPool(BoltServerAddress address) {
        return (ExtendedChannelPool)this.pools.get(address);
    }

    ExtendedChannelPool newPool(BoltServerAddress address) {
        return new NettyChannelPool(address, this.connector, this.bootstrap, this.nettyChannelTracker, this.channelHealthChecker, this.settings.connectionAcquisitionTimeout(), this.settings.maxConnectionPoolSize());
    }

    private ExtendedChannelPool getOrCreatePool(BoltServerAddress address) {
        return this.pools.computeIfAbsent(address, ignored -> {
            ExtendedChannelPool pool = this.newPool(address);
            this.metricsListener.putPoolMetrics(pool.id(), address, this);
            return pool;
        });
    }

    private CompletionStage<Void> closePool(ExtendedChannelPool pool) {
        return pool.close().whenComplete((ignored, error) -> this.metricsListener.removePoolMetrics(pool.id()));
    }

    private void closePoolInBackground(BoltServerAddress address, ExtendedChannelPool pool) {
        this.closePool(pool).whenComplete((ignored, error) -> {
            if (error != null) {
                this.log.warn(String.format("An error occurred while closing connection pool towards %s.", address), (Throwable)error);
            }
        });
    }

    private EventLoopGroup eventLoopGroup() {
        return this.bootstrap.config().group();
    }

    private void shutdownEventLoopGroup(Throwable pollCloseError) {
        this.eventLoopGroup().shutdownGracefully(200L, 15000L, TimeUnit.MILLISECONDS);
        Futures.asCompletionStage(this.eventLoopGroup().terminationFuture()).whenComplete((ignore, eventLoopGroupTerminationError) -> {
            CompletionException combinedErrors = Futures.combineErrors(pollCloseError, eventLoopGroupTerminationError);
            Futures.completeWithNullIfNoError(this.closeFuture, combinedErrors);
        });
    }

    private CompletableFuture<Void> closeAllPools() {
        return CompletableFuture.allOf((CompletableFuture[])this.pools.entrySet().stream().map(entry -> {
            BoltServerAddress address = (BoltServerAddress)entry.getKey();
            ExtendedChannelPool pool = (ExtendedChannelPool)entry.getValue();
            this.log.info("Closing connection pool towards %s", address);
            return this.closePool(pool).toCompletableFuture();
        }).toArray(CompletableFuture[]::new));
    }
}

