/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.driver.internal.pool;

import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import org.neo4j.driver.exceptions.Neo4jException;
import org.neo4j.driver.internal.pool.Allocator;
import org.neo4j.driver.internal.pool.Slot;
import org.neo4j.driver.internal.pool.ValidationStrategy;
import org.neo4j.driver.internal.util.Clock;
import org.neo4j.driver.internal.util.Consumer;

public class ThreadCachingPool<T>
implements AutoCloseable {
    private final ThreadLocal<Slot<T>> local = new ThreadLocal();
    private final BlockingQueue<Slot<T>> live = new LinkedBlockingQueue<Slot<T>>();
    private final BlockingQueue<Slot<T>> disposed = new LinkedBlockingQueue<Slot<T>>();
    private final Slot<T>[] all;
    private final int maxSize;
    private final AtomicInteger nextSlotIndex = new AtomicInteger(0);
    private final AtomicBoolean stopped = new AtomicBoolean(false);
    private final Allocator<T> allocator;
    private final ValidationStrategy<T> validationStrategy;
    private final Clock clock;

    public ThreadCachingPool(int targetSize, Allocator<T> allocator, ValidationStrategy<T> validationStrategy, Clock clock) {
        this.maxSize = targetSize;
        this.allocator = allocator;
        this.validationStrategy = validationStrategy;
        this.clock = clock;
        this.all = new Slot[targetSize];
    }

    public T acquire(long timeout, TimeUnit unit) throws InterruptedException {
        long deadline = this.clock.millis() + unit.toMillis(timeout);
        Slot<T> slot = this.local.get();
        if (slot != null && slot.availableToClaimed()) {
            if (slot.isValid(this.validationStrategy)) {
                this.allocator.onAcquire(slot.value);
                return slot.value;
            }
            this.dispose(slot);
        }
        return this.acquireFromGlobal(deadline);
    }

    private T acquireFromGlobal(long deadline) throws InterruptedException {
        Slot slot = (Slot)this.live.poll();
        while (true) {
            long timeLeft;
            if (this.stopped.get()) {
                throw new IllegalStateException("Pool has been closed, cannot acquire new values.");
            }
            if (slot != null) {
                if (slot.availableToClaimed()) {
                    if (slot.isValid(this.validationStrategy)) break;
                    this.dispose(slot);
                }
            } else {
                slot = (Slot)this.disposed.poll();
                if (slot != null) {
                    slot = this.allocate(slot.index);
                    break;
                }
                int index = this.nextSlotIndex.get();
                if (this.maxSize > index && this.nextSlotIndex.compareAndSet(index, index + 1)) {
                    slot = this.allocate(index);
                    break;
                }
            }
            if ((timeLeft = deadline - this.clock.millis()) <= 0L) {
                return null;
            }
            slot = this.live.poll(Math.min(timeLeft, 10L), TimeUnit.MILLISECONDS);
        }
        this.local.set(slot);
        this.allocator.onAcquire(slot.value);
        return slot.value;
    }

    private void dispose(Slot<T> slot) {
        if (!slot.claimedToDisposed()) {
            throw new IllegalStateException("Cannot dispose unclaimed pool object: " + slot);
        }
        this.disposed.add(slot);
        this.allocator.onDispose(slot.value);
    }

    private Slot<T> allocate(int slotIndex) {
        Slot<T> slot = new Slot<T>(slotIndex, this.clock);
        try {
            slot.set(this.allocator.allocate(this.createDisposeCallback(slot)));
            this.all[slotIndex] = slot;
            return slot;
        }
        catch (Neo4jException e) {
            slot.claimedToDisposed();
            this.disposed.add(slot);
            throw e;
        }
    }

    private Consumer<T> createDisposeCallback(final Slot<T> slot) {
        return new Consumer<T>(){

            @Override
            public void accept(T t) {
                slot.updateUsageTimestamp();
                if (!slot.isValid(ThreadCachingPool.this.validationStrategy)) {
                    ThreadCachingPool.this.dispose(slot);
                    return;
                }
                if (!slot.claimedToAvailable()) {
                    throw new IllegalStateException("Failed to release pooled object: " + slot);
                }
                if (!ThreadCachingPool.this.stopped.get()) {
                    ThreadCachingPool.this.live.add(slot);
                } else if (slot.availableToClaimed()) {
                    ThreadCachingPool.this.dispose(slot);
                }
            }
        };
    }

    @Override
    public void close() {
        if (!this.stopped.compareAndSet(false, true)) {
            return;
        }
        for (Slot<T> slot : this.all) {
            if (slot == null || !slot.availableToClaimed()) continue;
            this.dispose(slot);
        }
    }
}

