/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.shell.state;

import java.net.URI;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.logging.Level;
import java.util.logging.LogManager;
import org.neo4j.driver.AccessMode;
import org.neo4j.driver.AuthToken;
import org.neo4j.driver.AuthTokens;
import org.neo4j.driver.Bookmark;
import org.neo4j.driver.Config;
import org.neo4j.driver.Driver;
import org.neo4j.driver.GraphDatabase;
import org.neo4j.driver.Logging;
import org.neo4j.driver.Query;
import org.neo4j.driver.Record;
import org.neo4j.driver.Result;
import org.neo4j.driver.Session;
import org.neo4j.driver.SessionConfig;
import org.neo4j.driver.Transaction;
import org.neo4j.driver.TransactionConfig;
import org.neo4j.driver.Value;
import org.neo4j.driver.Values;
import org.neo4j.driver.exceptions.ClientException;
import org.neo4j.driver.exceptions.Neo4jException;
import org.neo4j.driver.exceptions.ServiceUnavailableException;
import org.neo4j.driver.exceptions.SessionExpiredException;
import org.neo4j.driver.internal.logging.DevNullLogging;
import org.neo4j.driver.summary.DatabaseInfo;
import org.neo4j.driver.summary.ResultSummary;
import org.neo4j.shell.ConnectionConfig;
import org.neo4j.shell.Connector;
import org.neo4j.shell.DatabaseManager;
import org.neo4j.shell.TransactionHandler;
import org.neo4j.shell.TriFunction;
import org.neo4j.shell.build.Build;
import org.neo4j.shell.exception.CommandException;
import org.neo4j.shell.exception.ThrowingAction;
import org.neo4j.shell.log.Logger;
import org.neo4j.shell.state.BoltResult;
import org.neo4j.shell.state.ErrorWhileInTransactionException;
import org.neo4j.shell.state.StatementBoltResult;
import org.neo4j.shell.state.TrialStatus;
import org.neo4j.shell.state.TrialStatusImpl;
import org.neo4j.shell.util.Versions;

/*
 * Multiple versions of this class in jar - see https://www.benf.org/other/cfr/multi-version-jar.html
 */
public class BoltStateHandler
implements TransactionHandler,
Connector,
DatabaseManager {
    private static final Logger log = Logger.create();
    private static final String USER_AGENT = "neo4j-cypher-shell/v" + Build.version();
    private static final TransactionConfig USER_DIRECT_TX_CONF = BoltStateHandler.txConfig(TransactionHandler.TransactionType.USER_DIRECT);
    private static final TransactionConfig SYSTEM_TX_CONF = BoltStateHandler.txConfig(TransactionHandler.TransactionType.SYSTEM);
    private final TriFunction<URI, AuthToken, Config, Driver> driverProvider;
    private final boolean isInteractive;
    private final Map<String, Bookmark> bookmarks = new HashMap<String, Bookmark>();
    protected Driver driver;
    Session session;
    private String protocolVersion;
    private String activeDatabaseNameAsSetByUser;
    private String actualDatabaseNameAsReportedByServer;
    private Transaction tx;
    private ConnectionConfig connectionConfig;
    private TrialStatus trialStatus = TrialStatusImpl.NOT_EXPIRED;

    public BoltStateHandler(boolean isInteractive) {
        this(GraphDatabase::driver, isInteractive);
    }

    BoltStateHandler(TriFunction<URI, AuthToken, Config, Driver> driverProvider, boolean isInteractive) {
        this.driverProvider = driverProvider;
        this.activeDatabaseNameAsSetByUser = "";
        this.isInteractive = isInteractive;
    }

    @Override
    public void setActiveDatabase(String databaseName) throws CommandException {
        if (this.isTransactionOpen()) {
            throw new CommandException("There is an open transaction. You need to close it before you can switch database.");
        }
        String previousDatabaseName = this.activeDatabaseNameAsSetByUser;
        this.activeDatabaseNameAsSetByUser = databaseName;
        try {
            if (this.isConnected()) {
                this.reconnectAndPing(databaseName, previousDatabaseName);
            }
        }
        catch (ClientException e) {
            if (this.isInteractive) {
                this.activeDatabaseNameAsSetByUser = previousDatabaseName;
                try {
                    this.reconnectAndPing(previousDatabaseName, previousDatabaseName);
                }
                catch (Exception e2) {
                    e.addSuppressed((Throwable)e2);
                }
            }
            throw e;
        }
    }

    @Override
    public String getActiveDatabaseAsSetByUser() {
        return this.activeDatabaseNameAsSetByUser;
    }

    @Override
    public String getActualDatabaseAsReportedByServer() {
        return this.actualDatabaseNameAsReportedByServer;
    }

    @Override
    public void beginTransaction() throws CommandException {
        if (!this.isConnected()) {
            throw new CommandException("Not connected to Neo4j");
        }
        if (this.isTransactionOpen()) {
            throw new CommandException("There is already an open transaction");
        }
        this.tx = this.session.beginTransaction(USER_DIRECT_TX_CONF);
    }

    @Override
    public void commitTransaction() throws CommandException {
        if (!this.isConnected()) {
            throw new CommandException("Not connected to Neo4j");
        }
        if (!this.isTransactionOpen()) {
            throw new CommandException("There is no open transaction to commit");
        }
        try {
            this.tx.commit();
            this.tx.close();
        }
        finally {
            this.tx = null;
        }
    }

    @Override
    public void rollbackTransaction() throws CommandException {
        if (!this.isConnected()) {
            throw new CommandException("Not connected to Neo4j");
        }
        if (!this.isTransactionOpen()) {
            throw new CommandException("There is no open transaction to rollback");
        }
        try {
            this.tx.rollback();
            this.tx.close();
        }
        finally {
            this.tx = null;
        }
    }

    public Neo4jException handleException(Neo4jException e) {
        if (this.isTransactionOpen()) {
            this.tx.close();
            this.tx = null;
            return new ErrorWhileInTransactionException("An error occurred while in an open transaction. The transaction will be rolled back and terminated. Error: " + e.getMessage(), e);
        }
        return e;
    }

    @Override
    public boolean isTransactionOpen() {
        return this.tx != null;
    }

    @Override
    public boolean isConnected() {
        return this.session != null && this.session.isOpen();
    }

    @Override
    public void connect(String user, String password, String database) throws CommandException {
        this.connect(this.connectionConfig.withUsernameAndPasswordAndDatabase(user, password, database));
    }

    @Override
    public void impersonate(String impersonatedUser) throws CommandException {
        if (this.isTransactionOpen()) {
            throw new CommandException("There is an open transaction. You need to close it before starting impersonation.");
        }
        if (this.isConnected()) {
            this.disconnect();
        }
        this.connect(this.connectionConfig.withImpersonatedUser(impersonatedUser));
    }

    @Override
    public void reconnect() throws CommandException {
        if (!this.isConnected()) {
            throw new CommandException("Can't reconnect when unconnected.");
        }
        if (this.isTransactionOpen()) {
            throw new CommandException("There is an open transaction. You need to close it before you can reconnect.");
        }
        ConnectionConfig config = this.connectionConfig;
        this.disconnect();
        this.connect(config);
    }

    @Override
    public void connect(ConnectionConfig incomingConfig) throws CommandException {
        if (this.isConnected()) {
            throw new CommandException("Already connected");
        }
        this.connectionConfig = BoltStateHandler.clean(incomingConfig);
        AuthToken authToken = AuthTokens.basic((String)this.connectionConfig.username(), (String)this.connectionConfig.password());
        try {
            String previousDatabaseName = this.activeDatabaseNameAsSetByUser;
            try {
                this.activeDatabaseNameAsSetByUser = this.connectionConfig.database();
                this.driver = this.getDriver(this.connectionConfig, authToken);
                this.reconnectAndPing(this.activeDatabaseNameAsSetByUser, previousDatabaseName);
            }
            catch (ServiceUnavailableException | SessionExpiredException e) {
                String fallbackScheme = switch (this.connectionConfig.uri().getScheme()) {
                    case "neo4j" -> "bolt";
                    case "neo4j+ssc" -> "bolt+ssc";
                    case "neo4j+s" -> "bolt+s";
                    default -> throw e;
                };
                this.connectionConfig = this.connectionConfig.withScheme(fallbackScheme);
                try {
                    this.driver = this.getDriver(this.connectionConfig, authToken);
                    log.info("Connecting with fallback scheme: " + fallbackScheme);
                    this.reconnectAndPing(this.activeDatabaseNameAsSetByUser, previousDatabaseName);
                }
                catch (Throwable fallbackThrowable) {
                    log.warn("Fallback scheme failed", fallbackThrowable);
                    throw e;
                }
            }
        }
        catch (Throwable t) {
            try {
                this.silentDisconnect();
            }
            catch (Exception e) {
                t.addSuppressed(e);
            }
            throw t;
        }
    }

    private void reconnectAndPing(String databaseToConnectTo, String previousDatabase) throws CommandException {
        this.reconnect(databaseToConnectTo, previousDatabase);
        this.getPing().apply();
        this.trialStatus = this.getTrialStatus();
    }

    private void reconnect(String databaseToConnectTo, String previousDatabase) {
        log.info("Connecting to database " + databaseToConnectTo + "...");
        SessionConfig.Builder builder = SessionConfig.builder();
        builder.withDefaultAccessMode(AccessMode.WRITE);
        if (!"".equals(databaseToConnectTo)) {
            builder.withDatabase(databaseToConnectTo);
        }
        this.closeSession(previousDatabase);
        Bookmark bookmarkForDBToConnectTo = this.bookmarks.get(databaseToConnectTo);
        if (bookmarkForDBToConnectTo != null) {
            builder.withBookmarks(new Bookmark[]{bookmarkForDBToConnectTo});
        }
        this.impersonatedUser().ifPresent(arg_0 -> ((SessionConfig.Builder)builder).withImpersonatedUser(arg_0));
        this.session = this.driver.session(builder.build());
        this.resetActualDbName();
    }

    private void closeSession(String databaseName) {
        if (this.session != null) {
            Bookmark bookmarkForPreviousDB = this.session.lastBookmark();
            this.session.close();
            this.bookmarks.put(databaseName, bookmarkForPreviousDB);
        }
    }

    private ThrowingAction<CommandException> getPing() {
        return () -> {
            try {
                Result run = this.session.run("CALL db.ping()", SYSTEM_TX_CONF);
                ResultSummary summary = run.consume();
                this.protocolVersion = summary.server().protocolVersion();
                this.updateActualDbName(summary);
            }
            catch (ClientException e) {
                log.warn("Ping failed", e);
                if (BoltStateHandler.procedureNotFound(e)) {
                    Result run = this.session.run(this.isSystemDb() ? "CALL db.indexes()" : "RETURN 1", SYSTEM_TX_CONF);
                    ResultSummary summary = run.consume();
                    this.protocolVersion = summary.server().protocolVersion();
                    this.updateActualDbName(summary);
                }
                throw e;
            }
        };
    }

    private TrialStatus getTrialStatus() {
        try {
            Record record = this.session.run("CALL dbms.acceptedLicenseAgreement()", SYSTEM_TX_CONF).single();
            return TrialStatus.parse(record.get(0).asString());
        }
        catch (Exception e) {
            log.warn("Failed to fetch trial status", e);
            return TrialStatusImpl.NOT_EXPIRED;
        }
    }

    @Override
    public String getServerVersion() {
        try {
            return this.runCypher("CALL dbms.components() YIELD versions", Collections.emptyMap(), SYSTEM_TX_CONF).flatMap(recordOpt -> recordOpt.getRecords().stream().findFirst()).map(record -> record.get("versions")).filter(value -> !value.isNull()).map(value -> value.get(0).asString()).orElse("");
        }
        catch (CommandException e) {
            log.warn("Failed to get server version", e);
            return "";
        }
    }

    @Override
    public String getProtocolVersion() {
        if (this.isConnected()) {
            if (this.protocolVersion == null) {
                this.protocolVersion = "";
            }
            return this.protocolVersion;
        }
        return "";
    }

    @Override
    public String username() {
        return this.connectionConfig != null ? this.connectionConfig.username() : "";
    }

    @Override
    public ConnectionConfig connectionConfig() {
        return this.connectionConfig;
    }

    @Override
    public Optional<String> impersonatedUser() {
        return Optional.ofNullable(this.connectionConfig).flatMap(ConnectionConfig::impersonatedUser);
    }

    @Override
    public Optional<BoltResult> runUserCypher(String cypher, Map<String, Object> queryParams) throws CommandException {
        return this.runCypher(cypher, queryParams, USER_DIRECT_TX_CONF);
    }

    @Override
    public Optional<BoltResult> runCypher(String cypher, Map<String, Object> queryParams, TransactionHandler.TransactionType type) throws CommandException {
        return this.runCypher(cypher, queryParams, BoltStateHandler.txConfig(type));
    }

    private Optional<BoltResult> runCypher(String cypher, Map<String, Object> queryParams, TransactionConfig config) throws CommandException {
        if (!this.isConnected()) {
            throw new CommandException("Not connected to Neo4j");
        }
        if (this.isTransactionOpen()) {
            return this.getBoltResult(cypher, queryParams, config);
        }
        try {
            return this.getBoltResult(cypher, queryParams, config);
        }
        catch (SessionExpiredException e) {
            log.warn("Failed to execute query, re-trying", e);
            this.reconnectAndPing(this.activeDatabaseNameAsSetByUser, this.activeDatabaseNameAsSetByUser);
            return this.getBoltResult(cypher, queryParams, config);
        }
    }

    public void updateActualDbName(ResultSummary resultSummary) {
        this.actualDatabaseNameAsReportedByServer = BoltStateHandler.getActualDbName(resultSummary);
    }

    public void changePassword(ConnectionConfig connectionConfig, String newPassword) {
        if (this.isConnected()) {
            this.silentDisconnect();
        }
        AuthToken authToken = AuthTokens.basic((String)connectionConfig.username(), (String)connectionConfig.password());
        try {
            this.driver = this.getDriver(connectionConfig, authToken);
            this.activeDatabaseNameAsSetByUser = "system";
            this.reconnect("system", "system");
            try {
                String command = "ALTER CURRENT USER SET PASSWORD FROM $o TO $n";
                Value parameters = Values.parameters((Object[])new Object[]{"o", connectionConfig.password(), "n", newPassword});
                Result run = this.session.run(new Query(command, parameters), BoltStateHandler.txConfig(TransactionHandler.TransactionType.USER_ACTION));
                run.consume();
            }
            catch (Neo4jException e) {
                if (Versions.isPasswordChangeRequiredException(e)) {
                    log.info("Password change failed, fallback to legacy method", e);
                    String oldCommand = "CALL dbms.security.changePassword($n)";
                    Value oldParameters = Values.parameters((Object[])new Object[]{"n", newPassword});
                    Result run = this.session.run(new Query(oldCommand, oldParameters), BoltStateHandler.txConfig(TransactionHandler.TransactionType.USER_ACTION));
                    run.consume();
                }
                throw e;
            }
            this.silentDisconnect();
        }
        catch (Throwable t) {
            try {
                this.silentDisconnect();
            }
            catch (Exception e) {
                t.addSuppressed(e);
            }
            if (t instanceof RuntimeException) {
                throw (RuntimeException)t;
            }
            throw new RuntimeException(t);
        }
    }

    private Optional<BoltResult> getBoltResult(String cypher, Map<String, Object> queryParams, TransactionConfig config) throws SessionExpiredException {
        Result statementResult = this.isTransactionOpen() ? this.tx.run(new Query(cypher, queryParams)) : this.session.run(new Query(cypher, queryParams), config);
        if (statementResult == null) {
            return Optional.empty();
        }
        return Optional.of(new StatementBoltResult(statementResult));
    }

    private static String getActualDbName(ResultSummary resultSummary) {
        DatabaseInfo dbInfo = resultSummary.database();
        return dbInfo.name() == null ? "" : dbInfo.name();
    }

    private void resetActualDbName() {
        this.actualDatabaseNameAsReportedByServer = null;
    }

    void silentDisconnect() {
        try {
            this.closeSession(this.activeDatabaseNameAsSetByUser);
            if (this.driver != null) {
                this.driver.close();
            }
        }
        finally {
            this.session = null;
            this.driver = null;
            this.resetActualDbName();
        }
    }

    public void reset() {
        if (this.isConnected() && this.isTransactionOpen()) {
            this.tx.rollback();
            this.tx = null;
        }
    }

    @Override
    public void disconnect() {
        this.reset();
        this.silentDisconnect();
        this.protocolVersion = null;
    }

    private Driver getDriver(ConnectionConfig connectionConfig, AuthToken authToken) {
        Config.ConfigBuilder configBuilder = Config.builder().withLogging(BoltStateHandler.driverLogger()).withUserAgent(USER_AGENT);
        switch (connectionConfig.encryption()) {
            case TRUE: {
                configBuilder = configBuilder.withEncryption();
                break;
            }
            case FALSE: {
                configBuilder = configBuilder.withoutEncryption();
                break;
            }
        }
        return this.driverProvider.apply(connectionConfig.uri(), authToken, configBuilder.build());
    }

    private boolean isSystemDb() {
        return this.activeDatabaseNameAsSetByUser.compareToIgnoreCase("system") == 0;
    }

    private static boolean procedureNotFound(ClientException e) {
        return "Neo.ClientError.Procedure.ProcedureNotFound".compareToIgnoreCase(e.code()) == 0;
    }

    private static Logging driverLogger() {
        Level level = LogManager.getLogManager().getLogger("").getLevel();
        if (level == Level.OFF) {
            return DevNullLogging.DEV_NULL_LOGGING;
        }
        return Logging.javaUtilLogging((Level)level);
    }

    private static ConnectionConfig clean(ConnectionConfig config) {
        if (config.impersonatedUser().filter(i -> i.equals(config.username())).isPresent()) {
            return config.withImpersonatedUser(null);
        }
        return config;
    }

    private static TransactionConfig txConfig(TransactionHandler.TransactionType type) {
        return TransactionConfig.builder().withMetadata(Map.of("type", type.value(), "app", "cypher-shell_v" + Build.version())).build();
    }

    public TrialStatus trialStatus() {
        return this.trialStatus;
    }
}

