package org.mule.weave.v2.module.xmlschema.utils

import org.apache.commons.io.IOUtils
import org.mule.apache.xerces.dom.DOMInputImpl
import org.mule.apache.xerces.impl.Constants.DISALLOW_DOCTYPE_DECL_FEATURE
import org.mule.apache.xerces.impl.Constants.XERCES_FEATURE_PREFIX
import org.mule.apache.xerces.impl.XMLEntityManager
import org.mule.apache.xerces.impl.dv.xs.XSSimpleTypeDecl
import org.mule.apache.xerces.impl.xs.SchemaSymbols
import org.mule.apache.xerces.impl.xs.XMLSchemaLoader.ENTITY_RESOLVER
import org.mule.apache.xerces.impl.xs.XSComplexTypeDecl
import org.mule.apache.xerces.impl.xs.opti.SchemaDOMParser
import org.mule.apache.xerces.impl.xs.opti.SchemaParsingConfig
import org.mule.apache.xerces.impl.xs.util.LSInputListImpl
import org.mule.apache.xerces.util.DOMEntityResolverWrapper
import org.mule.apache.xerces.util.DOMUtil
import org.mule.apache.xerces.util.URI.MalformedURIException
import org.mule.apache.xerces.xni.parser.XMLInputSource
import org.mule.apache.xerces.xs._
import org.mule.weave.v2.module.xmlschema.ResourceResolver.NO_NAMESPACE
import org.mule.weave.v2.module.xmlschema.StringUtils.isNotEmpty
import org.mule.weave.v2.parser.ast.types.UnionTypeNode
import org.mule.weave.v2.parser.ast.types.WeaveTypeNode
import org.w3c.dom.Element
import org.w3c.dom.Node
import org.w3c.dom.TypeInfo
import org.w3c.dom.ls.LSInput
import org.w3c.dom.ls.LSResourceResolver

import java.io.ByteArrayInputStream
import java.io.IOException
import java.io.InputStream
import java.io.StringReader
import java.nio.charset.StandardCharsets
import java.nio.charset.StandardCharsets.UTF_8
import java.util
import javax.xml.namespace.QName
import javax.xml.parsers.DocumentBuilderFactory
import scala.collection.mutable

object SchemaHelper {

  def getDocumentation(annotationContent: String): Seq[SchemaHelper.XmlDoc] = {
    var result: Seq[SchemaHelper.XmlDoc] = Seq()
    val dbFactory = DocumentBuilderFactory.newInstance
    try {
      val dBuilder = dbFactory.newDocumentBuilder
      val doc = dBuilder.parse(new ByteArrayInputStream(annotationContent.getBytes(UTF_8)))
      val children = doc.getDocumentElement.getChildNodes
      for (i <- 0 until children.getLength) {
        val child = children.item(i)
        if (isDocumentationTag(child)) {
          val textContent = child.getTextContent
          val attributes = child.getAttributes
          val lang = Option(attributes.getNamedItem("xml:lang")).map((attr: Node) => attr.getNodeValue)
          result = result :+ new SchemaHelper.XmlDoc(lang, textContent.trim)
        }
      }
    } catch {
      case e: Exception =>

      // todo: Ignore this exception???
    }
    result
  }

  def isDocumentationTag(child: Node): Boolean = child.isInstanceOf[Element] && getLocalName(child.asInstanceOf[Element]).endsWith("documentation")

  def getLocalName(child: Element): String = {
    val tagName = child.getTagName
    if (tagName.contains(":")) tagName.split(":")(1)
    else tagName
  }

  def buildUnionTypeNode[T](values: Seq[T], function: T => WeaveTypeNode): WeaveTypeNode = {
    UnionTypeNode(values.map(function.apply))
  }

  def getDefaultValue(element: XSElementDeclaration): Option[String] = {
    Option(element.getConstraintValue)
  }

  def getDefaultValue(attributeDeclaration: XSAttributeUse): Option[String] = {
    Option(attributeDeclaration.getConstraintValue)
  }

  def getFacet(element: XSSimpleTypeDecl, facetType: Int): XSObject = {
    if (facetType == XSSimpleTypeDefinition.FACET_ENUMERATION || facetType == XSSimpleTypeDefinition.FACET_PATTERN) {
      val list = element.getMultiValueFacets
      for (i <- 0 until list.getLength) {
        val f = list.item(i).asInstanceOf[XSMultiValueFacet]
        if (f.getFacetKind == facetType) {
          return f
        }
      }
    } else {
      val list = element.getFacets
      for (i <- 0 until list.getLength) {
        val f = list.item(i).asInstanceOf[XSFacet]
        if (f.getFacetKind == facetType) {
          return f
        }
      }
    }
    null
  }

  def getTargetNamespace(systemId: String, schema: String, domParser: SchemaDOMParser): String = {
    try domParser.parse(new XMLInputSource(null, systemId, null, new StringReader(schema), "UTF-8"))
    catch {
      case e: IOException =>
        //  TODO LOG ERROR
        return null
      case e: Exception =>
        return null
    }
    val document = domParser.getDocument
    val element = if (document != null) DOMUtil.getRoot(document) else null
    DOMUtil.getAttrValue(element, SchemaSymbols.ATT_TARGETNAMESPACE)
  }

  def getGroupType(complexType: XSComplexTypeDecl): Option[Short] = {
    val particle = complexType.getParticle
    if (particle != null) {
      val term = particle.getTerm
      getGroupType(term)
    } else {
      None
    }

  }

  def getGroupType(term: XSTerm): Option[Short] = {
    term match {
      case group: XSModelGroup => Some(group.getCompositor)
      case _                   => None
    }
  }

  def getTypeName(typeDefinition: XSTypeDefinition): Option[QName] = {
    typeDefinition match {
      case definition: TypeInfo =>
        if (definition.getTypeName != null) {
          Option(new QName(definition.getTypeNamespace, definition.getTypeName))
        } else {
          None
        }
      case _ => None
    }
  }

  def isRepeated(part: XSParticle) = part.getMaxOccursUnbounded || part.getMaxOccurs > 1 || part.getMinOccurs > 1

  def getSystemId(systemId: String): String = {
    try XMLEntityManager.expandSystemId(systemId, null, false)
    catch {
      case e: MalformedURIException =>
        systemId
    }
  }

  def getLSInputList(schemaByTargetNamespace: util.Map[String, util.List[DOMInputImpl]]): LSInputList = {
    val domInputs = new util.ArrayList[DOMInputImpl]()
    schemaByTargetNamespace.values.forEach(value => domInputs.addAll(value))
    val inputs = domInputs.toArray(new Array[LSInput](0))
    new LSInputListImpl(inputs, inputs.length)
  }

  @throws[IOException]
  def getSchemasByTargetNamespace(schemasMap: mutable.Map[String, InputStream], isDisallowDoctypeDeclarations: Boolean, resourceResolver: Option[LSResourceResolver]) = {
    val schemaDOMParser = new SchemaDOMParser(new SchemaParsingConfig())
    schemaDOMParser.setFeature(XERCES_FEATURE_PREFIX + DISALLOW_DOCTYPE_DECL_FEATURE, isDisallowDoctypeDeclarations)
    resourceResolver.foreach(resolver => schemaDOMParser.setProperty(ENTITY_RESOLVER, new DOMEntityResolverWrapper(resolver)))
    val schemaByTargetNamespace = new util.HashMap[String, util.List[DOMInputImpl]]()
    schemasMap.foreach((entry) => {
      val name = entry._1
      val inputStream = entry._2
      val schemaString = IOUtils.toString(inputStream, StandardCharsets.UTF_8)
      val domInput = new DOMInputImpl(null, getSystemId(name), null, schemaString, "UTF-8")
      val targetNamespace = getTargetNamespace(name, schemaString, schemaDOMParser)
      val key = if (isNotEmpty(targetNamespace)) targetNamespace else NO_NAMESPACE
      if (!schemaByTargetNamespace.containsKey(key)) schemaByTargetNamespace.put(key, new util.ArrayList[DOMInputImpl]())
      schemaByTargetNamespace.get(key).add(domInput)
    })
    schemaByTargetNamespace
  }

  class XmlDoc(var lang: Option[String], var content: String) {
    def getLang: Option[String] = lang

    def getContent: String = content
  }
}
