/**
* This module contains helper functions for working with URIs.
*
* To use this module, you must import it to your DataWeave code, for example,
* by adding the line `import * from dw::core::URL` to the header of your
* DataWeave script.
*/
%dw 2.0

/**
 * Encodes a URI with UTF-8 escape sequences.
 *
 *
 * Applies up to four escape sequences for characters composed of two "surrogate"
 * characters. The function assumes that the URI is a complete URI, so it does
 * not encode reserved characters that have special meaning.
 *
 * The function _does not encode these characters_ with UTF-8 escape sequences:
 *
 * [%header, cols="2,2"]
 * |===
 * | Type (not escaped)   | Examples
 * | Reserved characters  | ; , / ? : @ & = $
 * | Unescaped characters | alphabetic, decimal digits, - _ . ! ~ * ' ( )
 * | Number sign          | #
 * |===
 *
 * === Parameters
 *
 * [%header, cols="1,3"]
 * |===
 * | Name | Description
 * | text | The URI to encode.
 * |===
 *
 * === Example
 *
 * This example shows encodes spaces in one URL and lists some characters that
 * do not get encoded in the `not_encoded` string.
 *
 * ==== Source
 *
 * [source,DataWeave,linenums]
 * ----
 * %dw 2.0
 * import * from dw::core::URL
 * output application/json
 * ---
 * {
 *     "encodeURI" : encodeURI("http://asd/ text to decode /text"),
 *     "not_encoded": encodeURI("http://:;,/?:@&=\$_-_.!~*'()")
 * }
 * ----
 *
 * ==== Output
 *
 * [source,JSON,linenums]
 * ----
 * {
 *    "encodeURI": "http://asd/%20text%20to%20decode%20/%25/\"\'/text",
 *    "not_encoded": "http://:;,/?:@&=$_-_.!~*'()"
 * }
 * ----
 */
fun encodeURI(text: String): String = native("system::StringUrlEncodeFunctionValue")

/**
 * Decodes the escape sequences (such as `%20`) in a URI.
 *
 *
 * The function replaces each escape sequence in the encoded URI with the
 * character that it represents, but does not decode escape sequences that
 * could not have been introduced by `encodeURI`. The character `#` is not
 * decoded from escape sequences.
 *
 * === Parameters
 *
 * [%header, cols="1,3"]
 * |===
 * | Name | Description
 * | text | The URI to decode.
 * |===
 *
 * === Example
 *
 * This example decodes a URI that contains the URL percent encoding `%20`,
 * which is used for spaces.
 *
 * ==== Source
 *
 * [source,DataWeave,linenums]
 * ----
 * %dw 2.0
 * import * from dw::core::URL
 * output application/json
 * ---
 * {
 *   "decodeURI" : decodeURI('http://asd/%20text%20to%20decode%20/text')
 * }
 * ----
 *
 * ==== Output
 *
 * [source,JSON,linenums]
 * ----
 * {
 *   "decodeURI": "http://asd/ text to decode /text"
 * }
 * ----
 */
fun decodeURI(text: String): String = native("system::StringUrlDecodeFunctionValue")

/**
 * Escapes certain characters in a URI component using UTF-8 encoding.
 *
 *
 * There can be only four escape sequences for characters composed of two
 * "surrogate" * characters. `encodeURIComponent` escapes all characters
 * _except the following_: alphabetic, decimal digits, `- _ . ! ~ * ' ( )`.
 * Note that `encodeURIComponent` differs from `encodeURI` in that it encodes
 * reserved characters and the Number sign `#` of `encodeURI`:
 *
 * [%header, cols="2,2"]
 * |===
 * | Type                 | Includes
 * | Reserved characters  |
 * | Unescaped characters | alphabetic, decimal digits, - _ . ! ~ * ' ( )
 * | Number sign          |
 * |===
 *
 * === Parameters
 *
 * [%header, cols="1,3"]
 * |===
 * | Name | Description
 * | text | URI component string.
 * |===
 *
 * === Example
 *
 * This example encodes a variety of URI components.
 *
 * ==== Source
 *
 * [source,DataWeave,linenums]
 * ----
 * %dw 2.0
 * import * from dw::core::URL
 * output application/json
 * ---
 * {
 *   "comparing_encode_functions_output" : {
 *   	"encodeURIComponent" : encodeURI(" PATH/ TO /ENCODE "),
 *   	"encodeURI" : encodeURI(" PATH/ TO /ENCODE "),
 *   	"encodeURIComponent_to_hex" : encodeURIComponent(";,/?:@&="),
 *   	"encodeURI_not_to_hex" : encodeURI(";,/?:@&="),
 *   	"encodeURIComponent_not_encoded" : encodeURIComponent("-_.!~*'()"),
 *   	"encodeURI_not_encoded" : encodeURI("-_.!~*'()")
 *   }
 * }
 * ----
 *
 * ==== Output
 *
 * [source,JSON,linenums]
 * ----
 * {
 *   "comparing_encode_functions_output": {
 *     "encodeURIComponent": "%20PATH/%20TO%20/ENCODE%20",
 *     "encodeURI": "%20PATH/%20TO%20/ENCODE%20",
 *     "encodeURIComponent_to_hex": "%3B%2C%2F%3F%3A%40%26%3D",
 *     "encodeURI_not_to_hex": ";,/?:@&=",
 *     "encodeURIComponent_not_encoded": "-_.!~*'()",
 *     "encodeURI_not_encoded": "-_.!~*'()"
 *   }
 * }
 * ----
 */
fun encodeURIComponent(text: String): String = native("system::StringUrlEncodeComponentFunctionValue")

/**
 * Decodes a Uniform Resource Identifier (URI) component previously created
 * by `encodeURIComponent` or a similar routine.
 *
 *
 * For an example, see `encodeURIComponent`.
 *
 * === Parameters
 *
 * [%header, cols="1,3"]
 * |===
 * | Name | Description
 * | text | URI component string.
 * |===
 *
 * === Example
 *
 * This example decodes a variety of URI components.
 *
 * ==== Source
 *
 * [source,DataWeave,linenums]
 * ----
 * %dw 2.0
 * import * from dw::core::URL
 * output application/json
 * ---
 * {
 *   "decodeURIComponent": {
 *     "decodeURIComponent" : decodeURIComponent("%20PATH/%20TO%20/DECODE%20"),
 *     "decodeURIComponent" : decodeURIComponent("%3B%2C%2F%3F%3A%40%26%3D"),
 *     "decodeURIComponent" : decodeURIComponent("%2D%5F%2E%21%7E%2A%27%28%29%24"),
 *   }
 * }
 * ----
 *
 * ==== Output
 *
 * [source,JSON,linenums]
 * ----
 * {
 *    decodeURIComponent: {
 *      decodeURIComponent: " PATH/ TO /DECODE ",
 *      decodeURIComponent: ";,/?:@&=",
 *     decodeURIComponent: "-_.!~*'()\$"
 *    }
 * }
 * ----
 */
fun decodeURIComponent(text: String): String = native("system::StringUrlDecodeComponentFunctionValue")

// A type: URI.
/**
* Describes the URI type. For descriptions of the fields, see
* https://docs.mulesoft.com/dataweave/latest/dataweave-types#dw_type_url[URL Types (dw::core::URL)].
*/
type URI = {
  isValid: Boolean,
  host?: String,
  authority?: String,
  fragment?: String,
  path?: String,
  port?: Number,
  query?: String,
  scheme?: String,
  user?: String,
  isAbsolute?: Boolean,
  isOpaque?: Boolean
}

/**
 * Parses a URL and returns a `URI` object.
 *
 *
 * The `isValid: Boolean` property in the output `URI` object indicates whether
 * the parsing process succeeded. Every field in this object is optional, and
 * a field will appear in the output only if it was present in the URL input.
 *
 * === Parameters
 *
 * [%header, cols="1,3"]
 * |===
 * | Name | Description
 * | uri | The URI input.
 * |===
 *
 * === Example
 *
 * This example parses a URL.
 *
 * ==== Source
 *
 * [source,DataWeave,linenums]
 * ----
 * %dw 2.0
 * import * from dw::core::URL
 * output application/json
 * ---
 * {
 *   'composition': parseURI('https://en.wikipedia.org/wiki/Uniform_Resource_Identifier#footer')
 * }
 * ----
 *
 * ==== Output
 *
 * [source,JSON,linenums]
 * ----
 * {
 *   "composition": {
 *     "isValid": true,
 *     "raw": "https://en.wikipedia.org/wiki/Uniform_Resource_Identifier#footer",
 *     "host": "en.wikipedia.org",
 *     "authority": "en.wikipedia.org",
 *     "fragment": "footer",
 *     "path": "/wiki/Uniform_Resource_Identifier",
 *     "scheme": "https",
 *     "isAbsolute": true,
 *     "isOpaque": false
 *   }
 * }
 * ----
 */
fun parseURI(uri: String): URI = native("system::StringParseUrlFunctionValue")

/**
 * Uses a custom string interpolator to replace URL components with a
 * `encodeURIComponent` result. You can call this function using the standard call, or a simplified version.
 *
 * === Parameters
 *
 * [%header, cols="1,3"]
 * |===
 * | Name | Description
 * | baseStringArray | A string array that contains the URL parts to interpolate using the strings in the `interpolationStringArray`
 * parameter.
 * | interpolationStringArray | A string array that contains the strings used to interpolate the `baseStringArray`.
 * |===
 *
 * === Example
 *
 * The following example uses the compose function to form an encoded URL, the first parameter is an array of two strings that are part of the URL
 * and the second parameter is the `urlPath` variable that is used to interpolate the strings in the first parameter.
 * Notice that the spaces in the input are encoded in the output URL as `%20`.
 *
 * ==== Source
 *
 * [source,DataWeave,linenums]
 * ----
 * %dw 2.0
 * output application/json
 * var urlPath = "content folder"
 * import * from dw::core::URL
 * ---
 *  { "encodedURL" : compose(["http://examplewebsite.com/", "/page.html"], ["$(urlPath)"]) }
 * ----
 *
 * ==== Output
 *
 * [source,JSON,linenums]
 * ----
 * { "encodedURL" : "http://examplewebsite.com/content%20folder/page.html" }
 * ----
 *
 * === Example
 *
 * You can also call this function using the simplified syntax, which uses backticks (```) to enclose the string that includes the variable
 * to encode. This example shows how to use the simplified syntax to obtain the same result as in the previous example.
 *
 * ==== Source
 *
 * [source,DataWeave,linenums]
 * ----
 * %dw 2.0
 * output application/json
 * var urlPath = "content folder"
 * import * from dw::core::URL
 * ---
 * { "encodedURL" : compose `http://examplewebsite.com/$(urlPath)/page.html`}
 * ----
 *
 * ==== Output
 *
 * [source,JSON,linenums]
 * ----
 * { "encodedURL" : "http://examplewebsite.com/content%20folder/page.html" }
 * ----
 */
fun compose(parts: Array<String>, interpolation: Array<String>): String =
  parts[0] ++ (interpolation map (encodeURIComponent($) ++ parts[($$ + 1)]) joinBy '')
