package org.mule.weave.v2.runtime.core.operator.selectors

import org.mule.weave.v2.core.functions.BinaryFunctionValue
import org.mule.weave.v2.core.exception.ExecutionException
import org.mule.weave.v2.core.exception.IndexOutOfBoundsException
import org.mule.weave.v2.core.exception.InvalidSelectionException
import org.mule.weave.v2.core.exception.StrictSelectionOverNullSelection
import org.mule.weave.v2.core.io.SeekableStream
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.structure.ArraySeq
import org.mule.weave.v2.model.types._
import org.mule.weave.v2.model.values.ArrayValue
import org.mule.weave.v2.model.values.BinaryValue
import org.mule.weave.v2.model.values.CharSequenceValue
import org.mule.weave.v2.model.values.StringValue
import org.mule.weave.v2.model.values.Value
import org.mule.weave.v2.parser.location.WeaveLocation

class ArrayRangeSelectorOperator(override val location: WeaveLocation) extends BinaryFunctionValue {
  override val L = ArrayType

  override val R = RangeType

  override def doExecute(leftValue: L.V, rightValue: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    val arraySeq: ArraySeq = leftValue.evaluate
    val range = rightValue.evaluate
    val start = range.start
    val end = range.end
    try {
      val sliceResult = arraySeq.slice(start.toInt, end.toInt)
      ArrayValue(sliceResult, this)
    } catch {
      case e: ExecutionException => throw InvalidSelectionException(e)
    }
  }
}

class NullRangeSelectorOperator(override val location: WeaveLocation) extends BinaryFunctionValue {
  override val L = NullType

  override val R = RangeType

  override def doExecute(leftValue: L.V, rightValue: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    throw InvalidSelectionException(new StrictSelectionOverNullSelection(this))
  }
}
//TODO this method shouldn't allocate any ByteArray it should directly access the underlying SeekableStream
class BinaryRangeSelectorOperator(override val location: WeaveLocation) extends BinaryFunctionValue {
  override val L = BinaryType

  override val R = RangeType

  override def doExecute(leftValue: L.V, rightValue: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    val stream: SeekableStream = leftValue.evaluate
    val range = rightValue.evaluate
    val rangeStart = range.start.toLong
    val rangeEnd = range.end.toLong
    val end: Long = if (rangeEnd < 0) stream.size() + rangeEnd else rangeEnd
    val start: Long = if (rangeStart < 0) stream.size() + rangeStart else rangeStart
    val reverse = start > end
    if (reverse) {
      if (end < 0) {
        throw InvalidSelectionException(new IndexOutOfBoundsException(this.location, end, stream.size() - 1))
      }
      if (start >= stream.size()) {
        throw InvalidSelectionException(new IndexOutOfBoundsException(this.location, start, stream.size() - 1))
      }

      stream.seek(end)
      val length: Int = (start - end).asInstanceOf[Int] + 1
      val bytes: Array[Byte] = ctx.serviceManager.memoryService.newByteArray("SeekableStreamSourceReader", length)
      stream.read(bytes, 0, length)
      BinaryValue(bytes.reverse)
    } else {
      if (start < 0) {
        throw InvalidSelectionException(new IndexOutOfBoundsException(this.location, start, stream.size() - 1))
      }
      if (end >= stream.size()) {
        throw InvalidSelectionException(new IndexOutOfBoundsException(this.location, end, stream.size() - 1))
      }
      stream.seek(start)
      val length: Int = (end - start).asInstanceOf[Int] + 1
      val bytes = ctx.serviceManager.memoryService.newByteArray("SeekableStreamSourceReader", length)
      stream.read(bytes, 0, length)
      BinaryValue(bytes)
    }
  }
}

class StringRangeSelectorOperator(override val location: WeaveLocation) extends BinaryFunctionValue {
  override val L = StringType

  override val R = RangeType

  override def doExecute(leftValue: L.V, rightValue: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    var start = rightValue.evaluate.start.toInt
    var end = rightValue.evaluate.end.toInt
    val value: CharSequence = leftValue.evaluate
    end = if (end < 0) value.length + end else end
    start = if (start < 0) value.length + start else start
    val reverse = start > end
    if (reverse) {
      if (end < 0) {
        throw InvalidSelectionException(new IndexOutOfBoundsException(this.location, end, value.length - 1))
      }
      if (start >= value.length) {
        throw InvalidSelectionException(new IndexOutOfBoundsException(this.location, start, value.length - 1))
      }
      val slice: String = value.subSequence(end, start + 1).toString
      StringValue(slice.reverse, this)
    } else {
      if (start < 0) {
        throw InvalidSelectionException(new IndexOutOfBoundsException(this.location, start, value.length - 1))
      }
      if (end >= value.length) {
        throw InvalidSelectionException(new IndexOutOfBoundsException(this.location, end, value.length - 1))
      }
      CharSequenceValue(value.subSequence(start, end + 1), this)
    }
  }

}
