package org.mule.weave.v2.runtime.core.functions.mime

import org.mule.weave.v2.core.util.ObjectValueUtils.selectObject
import org.mule.weave.v2.core.util.ObjectValueUtils.selectString
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.structure.ObjectSeq
import org.mule.weave.v2.model.types.StringType
import org.mule.weave.v2.model.values.ObjectValue
import org.mule.weave.v2.model.values.ObjectValueBuilder
import org.mule.weave.v2.model.values.StringValue
import org.mule.weave.v2.parser.module.MimeType

object MimeTypeConverter {

  private val MAIN_TYPE_KEY = "type"
  private val SUB_TYPE_KEY = "subtype"
  private val PARAMETERS_KEY = "parameters"

  def toObjectValue(mimeType: MimeType): ObjectValue = {
    val builder = new ObjectValueBuilder()
    builder.addPair(MAIN_TYPE_KEY, StringValue(mimeType.mainType))
    builder.addPair(SUB_TYPE_KEY, StringValue(mimeType.subtype))

    val parametersBuilder = new ObjectValueBuilder()
    mimeType.parameters.foreach(p => {
      parametersBuilder.addPair(p._1, StringValue(p._2))
    })
    builder.addPair(PARAMETERS_KEY, parametersBuilder.build)
    builder.build
  }

  def toMimeType(mimeType: ObjectSeq)(implicit ctx: EvaluationContext): MimeType = {
    val mainType = selectString(mimeType, MAIN_TYPE_KEY).getOrElse("")
    val subtype = selectString(mimeType, SUB_TYPE_KEY).getOrElse("")
    val parameters = extractParameters(mimeType)
    MimeType(mainType, subtype, parameters)
  }

  private def extractParameters(mimeType: ObjectSeq)(implicit ctx: EvaluationContext): Map[String, String] = {
    var parameters = Map.empty[String, String]
    val parametersValue = selectObject(mimeType, PARAMETERS_KEY).getOrElse(ObjectSeq.empty)
    parametersValue.toSeq().foreach(kvp => {
      val name = kvp._1.evaluate.name
      val value = StringType.coerce(kvp._2).evaluate.toString
      parameters += (name -> value)
    })
    parameters
  }
}
