package org.mule.weave.v2.runtime.core.functions.collections

import org.mule.weave.v2.core.functions.BinaryFunctionValue
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.structure.KeyValuePair
import org.mule.weave.v2.model.structure.ObjectSeq
import org.mule.weave.v2.model.types.FunctionType
import org.mule.weave.v2.model.types.ObjectType
import org.mule.weave.v2.model.values._
import org.mule.weave.v2.model.values.math.Number

import scala.collection.mutable.ArrayBuffer

object MapObjectObjectFunctionValue extends BinaryFunctionValue {

  override val L: ObjectType = ObjectType

  override val R: FunctionType = FunctionType

  override def doExecute(leftValue: L.V, rightValue: R.V)(implicit ctx: EvaluationContext): Value[_] = {
    val objectSeq: ObjectSeq = leftValue.evaluate
    val valuePairs: Iterator[KeyValuePair] = objectSeq.toIterator()
    ObjectValue(new KeyValuePairFlattenIterator(new MapObjectIterator(valuePairs, rightValue)), this)
  }

  private def evaluateLambda(kvTuple: KeyValuePair, fn: FunctionValue, index: NumberValue)(implicit ctx: EvaluationContext): Iterator[KeyValuePair] = {
    val keyValue = kvTuple._1
    val value = kvTuple._2
    val res: Value[_] = fn.call(AttributeDelegateValue(value, keyValue), keyValue, index)
    val mapObjectLambdaObject = ObjectType.coerce(res, fn)
    mapObjectLambdaObject.evaluate(ctx).toIterator()
  }

  class MapObjectIterator(valuePairs: Iterator[KeyValuePair], rightValue: R.V)(implicit ctx: EvaluationContext) extends Iterator[Iterator[KeyValuePair]] {
    private var i = 0
    override def hasNext: Boolean = {
      valuePairs.hasNext
    }

    override def next(): Iterator[KeyValuePair] = {
      val keyValuePairs = evaluateLambda(valuePairs.next(), rightValue, NumberValue(Number(i)))
      i += 1
      keyValuePairs
    }
  }
}

/**
  * A Flatten over an iterator that does tail recursion elimination to avoid stack overflow on recursive functions that
  * Does mapObject
  * @param elements The key value pairs to be flatten
  */
class KeyValuePairFlattenIterator(val elements: Iterator[Iterator[KeyValuePair]]) extends Iterator[KeyValuePair] {

  private var iteratorProviders: ArrayBuffer[Iterator[Iterator[KeyValuePair]]] = ArrayBuffer(elements)
  private var nextValue: KeyValuePair = _
  private var nextValues: Iterator[KeyValuePair] = Iterator.empty

  def currentIterator(): Iterator[Iterator[KeyValuePair]] = {
    var lastOption = iteratorProviders.lastOption
    while (lastOption.isDefined && !lastOption.get.hasNext) {
      iteratorProviders.remove(iteratorProviders.length - 1)
      lastOption = iteratorProviders.lastOption
    }
    lastOption.getOrElse(Iterator.empty)
  }

  def loadNext(): Boolean = {
    var foundNext: Boolean = false
    var iterator = currentIterator()
    while (!foundNext && iterator.hasNext) {
      val theValue = iterator.next()
      theValue match {
        case fi: KeyValuePairFlattenIterator =>
          //avoid stacking FlattenIterators
          //Pull it up to avoid StackOverflow
          iteratorProviders.+=(fi.elements)
          nextValues = fi.nextValues
          if (fi.nextValue != null) {
            // if fi already loaded a value, then use it
            nextValue = fi.nextValue
            foundNext = true
          }
        case _ =>
          nextValues = theValue
          foundNext = nextValues.hasNext
      }

      if (!foundNext) {
        iterator = currentIterator()
      }
    }
    foundNext
  }

  override def hasNext: Boolean = {
    val hasNext = nextValue != null || nextValues.hasNext
    if (!hasNext) {
      loadNext()
    } else {
      true
    }
  }

  override def next(): KeyValuePair = {
    if (hasNext) {
      if (nextValue != null) {
        val result = nextValue
        nextValue = null
        result
      } else {
        nextValues.next()
      }
    } else {
      Iterator.empty.next()
    }
  }
}
