package org.mule.weave.v2.runtime.core.functions

import java.nio.charset.Charset
import org.mule.weave.v2.core.functions.TernaryFunctionValue
import org.mule.weave.v2.core.io.SeekableStream
import org.mule.weave.v2.core.util.CharsetUtil
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.structure.ObjectSeq
import org.mule.weave.v2.model.types.BinaryType
import org.mule.weave.v2.model.types.ObjectType
import org.mule.weave.v2.model.types.StringType
import org.mule.weave.v2.model.types.Type
import org.mule.weave.v2.model.types.UnionType
import org.mule.weave.v2.model.values._
import org.mule.weave.v2.module.DataFormatManager
import org.mule.weave.v2.module.reader.Reader
import org.mule.weave.v2.module.reader.SourceProvider
import org.mule.weave.v2.parser.location.LocationCapable
import org.mule.weave.v2.runtime.core.exception.ReadExecutionException

/**
  * Parses the content of the first argument using the specified mimeType and the specified reader properties.
  * The content. This parameter is mandatory
  * The mimeType of the content is optional, by default it's application/dw
  * A map with the reader properties is optional.
  */
class ReadFunctionValue extends TernaryFunctionValue {

  override val First: Type = UnionType(Seq(StringType, BinaryType))
  override val Second: Type = StringType
  override val Third: Type = ObjectType

  override val secondDefaultValue: Option[ValueProvider] = Some(ValueProvider(StringValue("application/dw")))
  override val thirdDefaultValue: Option[ValueProvider] = Some(ValueProvider(ObjectValue.empty))

  override def doExecute(firstValue: First.V, secondValue: Second.V, thirdValue: Third.V)(implicit ctx: EvaluationContext): Value[_] = {
    val firstArg: Value[Any] = firstValue
    val contentType: String = StringType.coerce(secondValue)(ctx).evaluate(ctx).toString
    val readerProperties: ObjectSeq = ObjectType.coerce(thirdValue).evaluate(ctx)
    val module = DataFormatManager.byContentTypeOrNameValue(StringType.coerce(secondValue))
    var content: StringType#T = ""
    val charset = firstArg.schema
      .flatMap((schema) => schema.encoding)
      .map((charsetName) => Charset.forName(charsetName))
      .getOrElse(CharsetUtil.defaultCharset)
    val reader: Reader =
      if (BinaryType.accepts(firstArg)) {
        content = "Binary Content"
        val binary: SeekableStream = BinaryType.coerce(firstArg).evaluate(ctx)
        ctx.registerCloseable(binary)
        module.reader(SourceProvider(binary, charset, module.defaultMimeType))
      } else {
        val st: StringType#T = StringType.coerce(firstArg).evaluate(ctx)
        content = st
        module.reader(createSourceProvider(st, firstArg, charset))
      }

    val keyValuePairs = readerProperties.toIterator()
    for (kvp <- keyValuePairs) {
      val propertyName = kvp._1.evaluate(ctx).name
      val value = StringType.coerce(kvp._2).evaluate(ctx)
      reader.setOption(kvp._1.location(), propertyName, value)
    }
    try {
      reader.read("root")
    } catch {
      case e: Exception =>
        throw ReadExecutionException(firstArg.location(), content, contentType, e.getMessage)
    }
  }

  protected def createSourceProvider(st: StringType#T, locatable: LocationCapable, charset: Charset)(implicit ctx: EvaluationContext): SourceProvider = {
    SourceProvider(st, charset)
  }

}
