package org.mule.weave.v2.runtime

import org.mule.weave.v2.core.functions.WriteFunctionValue
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.capabilities.AttributesCapable
import org.mule.weave.v2.model.structure.ArraySeq
import org.mule.weave.v2.model.structure.KeyValuePair
import org.mule.weave.v2.model.structure.ObjectSeq
import org.mule.weave.v2.model.structure.QualifiedName
import org.mule.weave.v2.model.types.ArrayType
import org.mule.weave.v2.model.types.BinaryType
import org.mule.weave.v2.model.types.BooleanType
import org.mule.weave.v2.model.types.DateTimeType
import org.mule.weave.v2.model.types.FunctionType
import org.mule.weave.v2.model.types.LocalDateTimeType
import org.mule.weave.v2.model.types.LocalDateType
import org.mule.weave.v2.model.types.LocalTimeType
import org.mule.weave.v2.model.types.NamespaceType
import org.mule.weave.v2.model.types.NullType
import org.mule.weave.v2.model.types.NumberType
import org.mule.weave.v2.model.types.ObjectType
import org.mule.weave.v2.model.types.PeriodType
import org.mule.weave.v2.model.types.RangeType
import org.mule.weave.v2.model.types.RegexType
import org.mule.weave.v2.model.types.StringType
import org.mule.weave.v2.model.types.TimeType
import org.mule.weave.v2.model.types.TimeZoneType
import org.mule.weave.v2.model.types.TypeType
import org.mule.weave.v2.model.values.KeyValue
import org.mule.weave.v2.model.values.NameValue
import org.mule.weave.v2.model.values.Value
import org.mule.weave.v2.parser.ast.QName

/**
  * Trait to be used as an outline for a given value
  */
sealed trait DataWeaveValue {

  /**
    * Returns the type of this value. Look At DataWeaveValueTypes for options
    * @return The types
    */
  def typeOf(): Int

  def typeName(): String

  /**
    * Returns the list of attributes
    * @return
    */
  def attributes(): Array[DataWeaveNameValue]

  /**
    * Selects the attribute with the given name and namepsace
    * @param localName The name
    * @param namespace The namespace
    * @return The value if any
    */
  def selectAttribute(localName: String, namespace: Option[String]): Option[DataWeaveValue]

  /**
    * Returns the array of all the schema properties
    *
    * Object and array schema properties are unsupported, it will always return an empty array.
    * @return The schema properties
    */
  def schemaProperties(): Array[SchemaValue] //TODO: review support for object and array

  /**
    * Selects the attribute with the given name and namepsace
    * @param localName The name
    * @return The value if any
    */
  def selectAttribute(localName: String): Option[DataWeaveValue] = {
    selectAttribute(localName, None)
  }

}

object DataWeaveValue {

  def apply(value: Value[_])(implicit ctx: EvaluationContext): DataWeaveValue = {
    DataWeaveValue(value, None)
  }

  def apply(value: KeyValuePair)(implicit ctx: EvaluationContext): DataWeaveValue = {
    DataWeaveValue(value._2, Some(value._1))
  }

  def apply(value: Value[_], keyValue: Option[Value[QualifiedName]])(implicit ctx: EvaluationContext): DataWeaveValue = {
    value match {
      case objectType if (ObjectType.accepts(objectType)) => {
        new ObjectDataWeaveValue(ObjectType.coerce(value), keyValue)
      }
      case arrayType if (ArrayType.accepts(arrayType)) => {
        new ArrayDataWeaveValue(ArrayType.coerce(value), keyValue)
      }
      case _ => {
        new SimpleDataWeaveValue(value, keyValue)
      }
    }
  }

  def attributes(key: Option[Value[QualifiedName]])(implicit ctx: EvaluationContext): Array[DataWeaveNameValue] = {
    key match {
      case Some(k: AttributesCapable) => {
        k.attributes
          .map((attrs) => {
            attrs.evaluate
              .toStream()
              .map((nv) => {
                val name = nv._1.evaluate
                DataWeaveNameValue(QName(name.name, name.namespace.map(_.uri)), DataWeaveValue(nv._2))
              })
              .toArray
          })
          .getOrElse(Array())

      }
      case _ => Array()
    }
  }

  def selectAttribute(key: Option[Value[QualifiedName]], localName: String, namespace: Option[String])(implicit ctx: EvaluationContext): Option[DataWeaveValue] = {
    key match {
      case Some(k: AttributesCapable) => {
        k.attributes
          .flatMap((attrs) => {
            attrs.evaluate
              .keyValueOf(NameValue(localName, namespace))
              .map((nvp) => DataWeaveValue(nvp._2))
          })
      }
      case _ => None
    }
  }
}

class ObjectDataWeaveValue(value: Value[ObjectSeq], key: Option[Value[QualifiedName]])(implicit ctx: EvaluationContext) extends DataWeaveValue {

  /**
    * Selects the value with the given name
    * @param localName The name to search
    * @return The value
    */
  def select(localName: String): Option[DataWeaveValue] = {
    val keyValuePair = value.evaluate.selectKeyValue(KeyValue(localName))
    if (keyValuePair == null) {
      None
    } else {
      Option(DataWeaveValue(keyValuePair))
    }
  }

  override def typeOf(): Int = {
    DataWeaveValueTypes.Object
  }

  /**
    * Selects the value with the given name
    * @param localName The name to search
    * @param namespace The namespace of the name
    * @return The value
    */
  def select(localName: String, namespace: String): Option[DataWeaveValue] = {
    val keyValuePair = value.evaluate.selectKeyValue(KeyValue(localName, namespace))
    if (keyValuePair == null) {
      None
    } else {
      Option(DataWeaveValue(keyValuePair))
    }
  }

  def entries(): Array[DataWeaveNameValue] = {
    value.evaluate
      .toSeq()
      .map((kvp) => {
        val key = kvp._1.evaluate
        DataWeaveNameValue(QName(key.name, key.namespace.map(_.uri)), DataWeaveValue(kvp))
      })
      .toArray
  }

  override def attributes(): Array[DataWeaveNameValue] = {
    DataWeaveValue.attributes(key)
  }

  override def selectAttribute(localName: String, namespace: Option[String]): Option[DataWeaveValue] = {
    DataWeaveValue.selectAttribute(key, localName, namespace)
  }

  override def typeName(): String = {
    value.valueType.name
  }

  override def schemaProperties(): Array[SchemaValue] = {
    value.schema
      .map(
        (s) =>
          s.properties()
            .map((p) => {
              SchemaValue(p.name.evaluate, p.value.evaluate)
            })
            .toArray)
      .getOrElse(Array())
  }
}

case class DataWeaveNameValue(name: QName, value: DataWeaveValue)

class ArrayDataWeaveValue(value: Value[ArraySeq], key: Option[Value[QualifiedName]])(implicit ctx: EvaluationContext) extends DataWeaveValue {

  /**
    * Returns the first element of the array if any
    * @return The value of the array if has any element
    */
  def head(): Option[DataWeaveValue] = {
    val option = value.evaluate.toSeq().headOption
    option.map(DataWeaveValue.apply)
  }
  override def typeOf(): Int = {
    DataWeaveValueTypes.Array
  }

  /**
    * Returns the type of the array
    * @return The type of the array if has any element
    */
  def elements(): Array[DataWeaveValue] = {
    val option = value.evaluate.toSeq()
    option.map(DataWeaveValue.apply).toArray
  }

  override def attributes(): Array[DataWeaveNameValue] = {
    DataWeaveValue.attributes(key)
  }

  override def selectAttribute(localName: String, namespace: Option[String]): Option[DataWeaveValue] = {
    DataWeaveValue.selectAttribute(key, localName, namespace)
  }

  override def typeName(): String = {
    value.valueType.name
  }

  override def schemaProperties(): Array[SchemaValue] = {
    value.schema
      .map(
        (s) =>
          s.properties()
            .map((p) => {
              SchemaValue(p.name.evaluate, p.value.evaluate)
            })
            .toArray)
      .getOrElse(Array())
  }
}

case class SchemaValue(name: String, value: Any)

class SimpleDataWeaveValue(value: Value[_], key: Option[Value[QualifiedName]])(implicit ctx: EvaluationContext) extends DataWeaveValue {

  /**
    * The value of this simple value
    * @return The value as String
    */
  def valueAsString(): String = {
    WriteFunctionValue.toDwString(value, ignoreSchema = true)
  }

  /**
    * The underling value
    * @return
    */
  def value(): Any = {
    value.evaluate
  }

  /**
    * Returns the list of schema properties
    * @return
    */
  def schemaProperties(): Array[SchemaValue] = {
    value.schema
      .map(
        (s) =>
          s.properties()
            .map((p) => {
              SchemaValue(p.name.evaluate, p.value.evaluate)
            })
            .toArray)
      .getOrElse(Array())
  }

  override def typeName(): String = {
    value.valueType.name
  }

  override def typeOf(): Int = {
    value.valueType.baseType match {
      case _: StringType     => DataWeaveValueTypes.String
      case BinaryType        => DataWeaveValueTypes.Binary
      case DateTimeType      => DataWeaveValueTypes.DateTime
      case LocalDateTimeType => DataWeaveValueTypes.LocalDateTime
      case TimeType          => DataWeaveValueTypes.Time
      case LocalTimeType     => DataWeaveValueTypes.LocalTime
      case PeriodType        => DataWeaveValueTypes.Period
      case TimeZoneType      => DataWeaveValueTypes.TimeZone
      case LocalDateType     => DataWeaveValueTypes.Date
      case _: ArrayType      => DataWeaveValueTypes.Array
      case _: BooleanType    => DataWeaveValueTypes.Boolean
      case NamespaceType(_)  => DataWeaveValueTypes.Namespace
      case _: NumberType     => DataWeaveValueTypes.Number
      case _: ObjectType     => DataWeaveValueTypes.Object
      case RangeType         => DataWeaveValueTypes.Range
      case RegexType         => DataWeaveValueTypes.Regex
      case _: FunctionType   => DataWeaveValueTypes.Function
      case NullType          => DataWeaveValueTypes.Null
      case TypeType          => DataWeaveValueTypes.Type
      case _                 => DataWeaveValueTypes.Unknown

    }
  }

  override def attributes(): Array[DataWeaveNameValue] = {
    DataWeaveValue.attributes(key)
  }

  override def selectAttribute(localName: String, namespace: Option[String]): Option[DataWeaveValue] = {
    DataWeaveValue.selectAttribute(key, localName, namespace)
  }
}

object DataWeaveValueTypes {
  val String = 0
  val Number = 1
  val Boolean = 2
  val Date = 3
  val DateTime = 4
  val LocalDateTime = 5
  val LocalTime = 6
  val Time = 7
  val Period = 8
  val Regex = 9
  val Range = 10
  val Namespace = 11
  val Type = 12
  val Binary = 13
  val TimeZone = 14
  val Array = 15
  val Object = 16
  val Null = 17
  val Function = 18
  val Unknown = 1000
}
