package org.mule.weave.v2.interpreted.node.structure

import org.mule.weave.v2.interpreted.ExecutionContext
import org.mule.weave.v2.interpreted.node.ValueNode
import org.mule.weave.v2.model.structure.ArraySeq
import org.mule.weave.v2.model.values.coercion.ArrayCoercer
import org.mule.weave.v2.model.values.ArrayValue
import org.mule.weave.v2.model.values._

class DynamicArrayNode(val v: Seq[ValueNode[_]]) extends ValueNode[ArraySeq] {

  override def productElement(n: Int): Any = v.apply(n)

  override def productArity: Int = v.length

  override def doExecute(implicit ctx: ExecutionContext): Value[ArraySeq] = {
    val filtered = v
      .filter({
        case cond: ConditionalCapableNode => cond.condition
        case _                            => true
      })
      .map(_.execute)
    ArrayValue(filtered, this)
  }
}

class ArrayNode(val v: Array[ValueNode[_]]) extends ValueNode[ArraySeq] {

  override def productElement(n: Int): Any = v.apply(n)

  override def productArity: Int = v.length

  override def doExecute(implicit ctx: ExecutionContext): Value[ArraySeq] = {
    val result = new Array[Value[_]](v.length)
    var i = 0
    while (i < v.length) {
      result.update(i, v(i).execute)
      i = i + 1
    }
    ArrayValue(ArraySeq(result), this)
  }
}

class LiteralArrayNode(val elements: Array[ValueNode[_]]) extends ValueNode[ArraySeq] {

  var value: Value[ArraySeq] = _

  override def productElement(n: Int): Any = elements.apply(n)

  override def productArity: Int = elements.length

  override def doExecute(implicit ctx: ExecutionContext): Value[ArraySeq] = {
    if (value == null) {
      synchronized {
        if (value == null) {
          val result = new Array[Value[_]](elements.length)
          var i = 0
          while (i < elements.length) {
            result.update(i, elements(i).execute)
            i = i + 1
          }
          value = ArrayValue(ArraySeq(result, materialized = true), this)
        }
      }
    }
    value
  }
}

class HeadTailArrayNode(val head: ValueNode[_], val tail: ValueNode[_]) extends ValueNode[ArraySeq] {

  override def productElement(n: Int): Any = n match {
    case 0 => head
    case 1 => tail
  }

  override def productArity: Int = 2

  override def doExecute(implicit ctx: ExecutionContext): Value[ArraySeq] = {
    val frame = ctx.executionStack().activeFrame()
    val valueProducer = () => {
      ctx.runInFrame(frame, ArrayCoercer.coerceToArraySeq(tail.execute, this).toIterator())
    }
    head match {
      case cond: ConditionalCapableNode => {
        if (cond.condition) {
          val headTailArraySeq = new HeadTailValueIterator(head.execute, valueProducer)
          ArrayValue(headTailArraySeq, this)
        } else {
          ArrayValue(valueProducer.apply(), this)
        }
      }
      case _ => {
        val headTailArraySeq = new HeadTailValueIterator(head.execute, valueProducer)
        ArrayValue(headTailArraySeq, this)
      }
    }

  }
}

//tail is a value producer
class HeadTailValueIterator(var head: Value[_], var tail: () => Iterator[Value[_]]) extends Iterator[Value[_]] {

  def consumeTail(): Unit = {
    val tailArraySeq: Iterator[Value[_]] = tail()
    tailArraySeq match {
      case ht: HeadTailValueIterator if ht.hasNext =>
        head = ht.head
        tail = ht.tail
      case seq: Iterator[Value[_]] if seq.hasNext =>
        head = seq.next()
        tail = () => seq
      case _ =>
    }
  }

  override def hasNext: Boolean = {
    if (head == null) {
      consumeTail()
    }
    head != null
  }

  override def next(): Value[_] = {
    val result: Value[_] = head
    head = null
    result
  }
}
