package org.mule.weave.v2.interpreted.node

import org.mule.weave.v2.core.exception.InternalExecutionException
import org.mule.weave.v2.interpreted.ExecutionContext
import org.mule.weave.v2.interpreted.listener.NotificationManager
import org.mule.weave.v2.model.values.Value

/**
  * All the AstNodes that returns/generates a data weave value
  *
  * @tparam T The type of value that generates
  */
trait ValueNode[+T] extends ExecutionNode {

  /**
    * Executes the node and returns the result value of evaluating
    *
    * @param ctx The evaluation context
    * @return The result value
    */
  final def execute(implicit ctx: ExecutionContext): Value[T] = {
    val manager: NotificationManager = ctx.notificationManager()
    if (manager.nonEmpty && shouldNotify) {
      executeWithNotifications(manager)
    } else {
      doExecute
    }
  }

  private def executeWithNotifications(manager: NotificationManager)(implicit ctx: ExecutionContext) = {
    manager.preValueNodeExecution(this)
    try {
      var result: Value[T] = doExecute
      if (ctx.materializeValues()) {
        result = result.materialize
      }
      manager.postValueNodeExecution(this, result)
      result
    } catch {
      case e: InternalExecutionException => {
        throw e
      }
      case e: Exception =>
        manager.postValueNodeExecution(this, e)
        throw e
    }
  }

  /**
    * If it returns false it will not generate any notification. This  means that this node is not important
    * enough to be measure
    * @return True if it will generate notifications
    */
  def shouldNotify: Boolean = true

  /**
    * Execute methods that does the real magic
    * @param ctx The execution context
    * @return The new value
    */
  protected def doExecute(implicit ctx: ExecutionContext): Value[T]

}
