package org.mule.weave.v2.interpreted.node

import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.types.Type
import org.mule.weave.v2.model.types.Types
import org.mule.weave.v2.model.values.FunctionParameter
import org.mule.weave.v2.model.values.FunctionValue
import org.mule.weave.v2.model.values.Value
import org.mule.weave.v2.interpreted.ExecutionContext
import org.mule.weave.v2.parser.ast.WeaveLocationCapable

import java.util
import scala.collection.mutable

object FunctionDispatchingHelper {

  def allTargets(functionValue: FunctionValue)(implicit ctx: EvaluationContext): Array[_ <: FunctionValue] = {
    if (functionValue.isOverloaded) {
      functionValue.overloads
    } else {
      val result = new Array[FunctionValue](1)
      result.update(0, functionValue)
      result
    }
  }

  /**
    * Finds the method to call
    */
  def findMatchingFunction(arguments: Array[Value[Any]], options: Array[_ <: FunctionValue])(implicit ctx: EvaluationContext): Option[(Int, FunctionValue)] = {
    var optionIndex: Int = 0
    while (optionIndex < options.length) {
      val functionValue: FunctionValue = options(optionIndex)
      val matchesTypes = matchesFunctionTypes(functionValue, arguments)
      if (matchesTypes) {
        return Some(optionIndex, functionValue)
      }
      optionIndex = optionIndex + 1
    }
    None
  }

  /**
    * Returns the function value that can be executed with this arguments after coercing. It allows provides the coerced values and the index of the ones that need to be coerced
    *
    * @param arguments The arguments that we are trying to use
    * @param options   All the functions options
    * @param ctx       The execution context
    * @return
    */
  def findMatchingFunctionWithCoercion(arguments: Array[Value[Any]], options: Array[_ <: FunctionValue], weaveLocationCapable: WeaveLocationCapable)(implicit ctx: EvaluationContext): Option[(Int, Array[Value[_]], Seq[Int])] = {
    var optionIndex: Int = 0
    val coercedValues = new Array[Value[_]](arguments.length)
    while (optionIndex < options.length) {

      val function: FunctionValue = options(optionIndex)
      if (function.minParams <= arguments.length && function.maxParams >= arguments.length) {
        var index: Int = 0
        var matchesTypes = true
        val functionParamTypes: Array[Type] = function.parameterTypes
        val paramsThatAreCoerced = new mutable.ArrayBuffer[Int]()
        //We already check arity now we only validate and try to coerce the specified parameters
        while (arguments.length > index && matchesTypes) {
          val expectedType: Type = functionParamTypes(index)
          if (!expectedType.accepts(arguments(index))) {
            try {
              coercedValues.update(index, expectedType.coerce(arguments(index), weaveLocationCapable))
              paramsThatAreCoerced.+=(index)
            } catch {
              case _: Exception => matchesTypes = false
            }
          } else {
            coercedValues.update(index, arguments(index))
          }
          index = index + 1
        }
        if (matchesTypes) {
          return Some((optionIndex, coercedValues, paramsThatAreCoerced))
        }
      }
      optionIndex = optionIndex + 1
    }
    None
  }

  def indexOfFunction(options: Array[_ <: FunctionValue], possible: FunctionValue): Int = {
    var i = 0
    while (i < options.length) {
      if (options(i) eq possible) {
        return i
      }
      i = i + 1
    }
    -1
  }

  def materializeArgs(parameters: Array[FunctionParameter], arguments: Array[Value[Any]])(implicit ctx: ExecutionContext): Array[Value[Any]] = {
    val argsIterator = arguments.iterator
    var index = 0
    val result = new Array[Value[Any]](arguments.length)
    while (argsIterator.hasNext) {
      val arg = argsIterator.next()
      val newArg = if (parameters(index).typeRequiresMaterialization) {
        arg.materialize
      } else {
        arg
      }
      result.update(index, newArg)
      index = index + 1
    }
    result
  }

  /**
    * This method return the arguments materialized if the corresponding parameters require that to check if the types match.
    * This is for Objects with defined fields, for example.
    */
  def materializeOverloadedFunctionArgs(overloads: Array[_ <: FunctionValue], arguments: Array[Value[Any]])(implicit ctx: EvaluationContext): Array[Value[Any]] = {
    var index = 0
    val result = new Array[Value[Any]](arguments.length)
    while (index < arguments.length) {
      val arg: Value[Any] = arguments(index)
      val newArg = materializeOverloadedFunctionArgs(overloads, index, arg)
      result.update(index, newArg)
      index = index + 1
    }
    result
  }

  def materializeOverloadedFunctionArgs(overloads: Array[_ <: FunctionValue], argumentIndex: Int, argument: Value[Any])(implicit ctx: EvaluationContext): Value[Any] = {
    val requiresMaterialize: Boolean = overloads.exists(fun => {
      if (fun.parameters.length > argumentIndex) {
        fun.parameters(argumentIndex).typeRequiresMaterialization
      } else {
        false
      }
    })
    val newArg = if (requiresMaterialize) {
      argument.materialize
    } else {
      argument
    }
    newArg
  }

  def sortByParameterTypeWeight(functionOverloads: Array[_ <: FunctionValue], argTypes: Array[Type])(implicit ctx: EvaluationContext): Array[FunctionValue] = {
    val weightFunctions: Array[(FunctionValue, Double, Int)] = functionOverloads.map((operator) => {
      val parameterTypes: Array[Type] = operator.parameters.map(_.wtype)
      var weight: Double = 0
      var miss = 0
      parameterTypes
        .zip(argTypes)
        .foreach((paramTypeArgType) => {
          if (!(paramTypeArgType._2.baseType eq paramTypeArgType._1.baseType)) {
            miss = miss + 1
            weight = weight + paramTypeArgType._1.weight
          }
        })
      (operator, weight, miss)
    })

    /**
      * This algorithm sorts the operators based on its types.
      * It will sort by the amount of type hits and then, the ones with the same amount of hits will be sorted by distance based on the type weights.
      * The weight is a number that is bigger based on how generic a type is. The more generic, bigger the number.
      */
    val sortedOperators = weightFunctions
      .sortBy((v) => v._2)
      .sortBy((v) => v._3)
      .map((v) => v._1)
    sortedOperators
  }

  def tryToCoerceOnly(args: Array[Value[_]], dispatch: FunctionValue, paramIndexToCoerce: Seq[Int])(implicit ctx: EvaluationContext): Array[Value[_]] = {
    val result = args.clone()
    val parameterTypes = dispatch.parameterTypes
    var i = 0
    while (paramIndexToCoerce.length > i) {
      val paramIndex = paramIndexToCoerce(i)
      val arg = args(paramIndex)
      val expectedType = parameterTypes(paramIndex)
      val valueMaybe = expectedType.coerceMaybe(arg)
      if (valueMaybe.isDefined) {
        result.update(paramIndex, valueMaybe.get)
      } else {
        return null
      }
      i = i + 1
    }
    result
  }

  def tryToCoerce(argumentsWithDefaults: Array[Value[_]], dispatch: FunctionValue)(implicit ctx: EvaluationContext): Option[Array[Value[_]]] = {
    var i = 0
    val result = new Array[Value[_]](dispatch.parameters.length)
    while (dispatch.parameters.length > i) {
      val expectedType = dispatch.parameters(i).wtype
      val value = argumentsWithDefaults(i)
      if (!expectedType.accepts(value)) {
        val valueMaybe = expectedType.coerceMaybe(value)
        if (valueMaybe.isDefined) {
          result.update(i, valueMaybe.get)
        } else {
          return None
        }
      } else {
        result.update(i, value)
      }
      i = i + 1
    }
    Some(result)
  }

  def matchesFunctionTypes(functionValue: FunctionValue, arguments: Array[Value[Any]])(implicit ctx: EvaluationContext): Boolean = {
    matchesParameters(functionValue.parameters, functionValue.parameterTypes, arguments)
  }

  def matchesParameters(parameters: Array[FunctionParameter], paramTypes: Array[Type], arguments: Array[Value[Any]])(implicit ctx: EvaluationContext): Boolean = {
    var matchesTypes = true
    if (parameters.length == arguments.length) {
      matchesTypes = Types.validate(paramTypes, arguments)
    } else if (arguments.length > parameters.length) {
      matchesTypes = false
    } else {
      //Head default parameters
      if (parameters.nonEmpty && parameters.head.value.nonEmpty) {
        val delta: Int = parameters.length - arguments.length
        val requireDefaultValue = util.Arrays.copyOfRange(parameters, 0, delta)
        val needsToCheck = util.Arrays.copyOfRange(parameters, delta, parameters.length)
        val needsToCheckTypes = util.Arrays.copyOfRange(paramTypes, delta, parameters.length)

        matchesTypes = validateDefaultValue(requireDefaultValue) && matchesParameters(needsToCheck, needsToCheckTypes, arguments)
      } else if (parameters.nonEmpty && parameters.last.value.nonEmpty) {
        //Tail parameters
        val delta: Int = parameters.length - arguments.length
        val requireDefaultValue = util.Arrays.copyOfRange(parameters, parameters.length - delta, parameters.length)
        val needsToCheck = util.Arrays.copyOfRange(parameters, 0, parameters.length - delta)
        val needsToCheckTypes = util.Arrays.copyOfRange(paramTypes, 0, parameters.length - delta)
        matchesTypes = validateDefaultValue(requireDefaultValue) && matchesParameters(needsToCheck, needsToCheckTypes, arguments)
      } else {
        matchesTypes = false
      }
    }
    matchesTypes
  }

  private def validateDefaultValue(requireDefaultValue: Array[FunctionParameter]): Boolean = {
    var i = 0
    while (i < requireDefaultValue.length) {
      if (requireDefaultValue(i).value.isEmpty) {
        return false
      }
      i = i + 1
    }
    true
  }

  final def expandArguments(arguments: Array[Value[_]], function: FunctionValue)(implicit ctx: EvaluationContext): Array[Value[_]] = {
    FunctionValue.expandArguments(arguments, function)
  }

}
