package org.mule.weave.v2.interpreted.module

import org.mule.weave.v2.core.util.BinaryHelper

import java.io.OutputStream
import java.nio.charset.StandardCharsets

import org.mule.weave.v2.interpreted.module.reader.OnlyDataSettings
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.structure.ArraySeq
import org.mule.weave.v2.model.structure.Namespace
import org.mule.weave.v2.model.structure.ObjectSeq
import org.mule.weave.v2.model.structure.QualifiedName
import org.mule.weave.v2.model.structure._
import org.mule.weave.v2.model.structure.schema.Schema
import org.mule.weave.v2.model.structure.schema.SchemaProperty
import org.mule.weave.v2.model.types._
import org.mule.weave.v2.model.values._
import org.mule.weave.v2.model.values.helper.AttributeHelper
import org.mule.weave.v2.module.DataFormat
import org.mule.weave.v2.module.option.BooleanModuleOption
import org.mule.weave.v2.module.option.IntModuleOption
import org.mule.weave.v2.module.option.ModuleOption
import org.mule.weave.v2.module.option.StringModuleOption
import org.mule.weave.v2.module.reader.ConfigurableDeferred
import org.mule.weave.v2.module.writer.BufferedIOWriter
import org.mule.weave.v2.module.writer.TargetProvider
import org.mule.weave.v2.module.writer.Writer
import org.mule.weave.v2.parser.location.LocationCapable
import org.mule.weave.v2.utils.StringEscapeHelper

import scala.collection.mutable

class WeaveWriter(os: OutputStream, val settings: WeaveWriterSettings)(implicit ctx: EvaluationContext) extends Writer {

  private lazy val writer: BufferedIOWriter = {
    BufferedIOWriter(os, StandardCharsets.UTF_8, settings.bufferSize)
  }
  private val declaredNs = mutable.Stack[mutable.ArrayBuffer[Namespace]]()

  private var insideHeader = false

  private var indent: Int = 0

  override def supportsStreaming: Boolean = false

  override def close(): Unit = {
    writer.close()
  }

  def doIndent(): Unit = {
    indent = indent + 1
  }

  def doDedent(): Unit = {
    indent = indent - 1
  }

  override def startDocument(location: LocationCapable): Unit = {
    declaredNs.push(mutable.ArrayBuffer())
  }

  def startHeaderIfRequired(): Unit = {
    if (!insideHeader) {
      writer.write("%dw 2.0")
      newline()
      insideHeader = true
    }
  }

  override def defineNamespace(location: LocationCapable, prefix: String, uri: String): Unit = {
    startHeaderIfRequired()
    writeNamespace(prefix, uri)
  }

  def getPrefixFor(prefix: String, uri: String): String = {
    val mayBeNamespace = declaredNs.toStream.flatMap(_.filter(_.uri == uri)).headOption
    mayBeNamespace match {
      case Some(value) => {
        value.prefix
      }
      case None => {
        var resultPrefix = prefix
        if (prefix.isEmpty) {
          resultPrefix = s"ns${declaredNs.top.size}"
        }
        var alreadyDefined = true
        var i = 0
        var tmpPrefix = resultPrefix
        while (alreadyDefined) {
          if (declaredNs.top.exists(_.prefix == tmpPrefix)) {
            tmpPrefix = s"${resultPrefix}${i}"
          } else {
            alreadyDefined = false
            resultPrefix = tmpPrefix
          }
          i = i + 1
        }
        resultPrefix
      }
    }
  }

  private def writeNamespace(prefix: String, uri: String): Unit = {
    if (uri.nonEmpty) {
      if (!declaredNs.top.exists(_.prefix == prefix)) {
        writer.write("ns ")
        writer.write(prefix)
        writer.write(" ")
        writer.write(uri)
        declaredNs.top.+=(Namespace(prefix, uri))
        newline()
      }
    }
  }

  private def startAttributes(location: LocationCapable): Unit = {
    writer.write(" @(")
  }

  private def endAttributes(location: LocationCapable): Unit = {
    writer.write(")")
  }

  private def startObject()(implicit ctx: EvaluationContext): Unit = {
    writer.write("{")
    doIndent()
    newline()
  }

  private def endObject(): Unit = {
    doDedent()
    newline()
    writer.write("}")
  }

  private def startArray(location: LocationCapable)(implicit ctx: EvaluationContext): Unit = {
    writer.write("[")
    doIndent()
    newline()
  }

  private def endArray(location: LocationCapable): Unit = {
    doDedent()
    newline()
    writer.write("]")
  }

  def requiresQuotes(key: String): Boolean = {
    StringEscapeHelper.keyRequiresQuotes(key)
  }

  private def newline(): Unit = {
    if (settings.indent.length > 0) {
      writer.write("\n")
      if (indent > 0) {
        writer.write(settings.indent * indent)
      }
    }
  }

  override def result: OutputStream = {
    writer.flush()
    os
  }

  def writeSchema(t: Type, schema: Option[Schema])(implicit ctx: EvaluationContext): Unit = {
    schema match {
      case Some(theSchema) if theSchema.properties().nonEmpty =>
        writer.write(" as ")
        //We write only base types as we can not reference to types that we are not persisting
        writer.write(t.baseType.name)
        writer.write(" {")
        var first = true
        theSchema.properties().foreach((sproperty) => {
          if (!sproperty.internal) {
            if (!first) {
              writer.append(", ")
            }
            writer.append(sproperty.name.evaluate)
            writer.write(": ")
            writeValue(sproperty.value)
            first = false
          }
        })
        writer.write("}")
        writer.flush()
      case _ =>
    }
  }

  def qualifiedName(key: Value[QualifiedName], attributes: Option[Value[NameSeq]])(implicit ctx: EvaluationContext): Unit = {
    val evaluate = key.evaluate
    evaluate.namespace match {
      case Some(ns) if (ns.uri.nonEmpty) => writer.write(getUniquePrefix(ns) + "#")
      case _                             =>
    }
    if (requiresQuotes(evaluate.name)) {
      writeQuoteString(evaluate.name)
    } else {
      writer.write(evaluate.name)
    }

    attributes match {
      case Some(attrs) => {
        val attributes = attrs.evaluate
        if (!attributes.isEmpty) {
          startAttributes(attrs)
          var first = true
          attributes.toIterator().foreach((nvp) => {
            if (!first) {
              writer.write(", ")
            }
            qualifiedName(nvp._1, None)
            writer.write(": ")
            writeValue(nvp._2)
            first = false
          })
          endAttributes(attrs)
        }
      }
      case None =>
    }
  }

  def alreadyDefined(ns: Namespace): Boolean = {
    declaredNs.exists((context) => {
      context.exists(_.uri == ns.uri)
    })
  }

  def writeDocumentSeparatorIfRequired(): Unit = {
    if (insideHeader) {
      writer.write("---")
      newline()
      insideHeader = false
    }
  }

  protected override def doWriteValue(v: Value[_])(implicit ctx: EvaluationContext): Unit = {
    var ignoreSchema = settings.ignoreSchema
    writeDocumentSeparatorIfRequired()
    v match {
      case t if BooleanType.accepts(t) =>
        writer.write(v.evaluate.toString)
      case t if NumberType.accepts(t) =>
        writer.write(NumberType.coerce(v).evaluate.toCanonicalString)
      case t if NullType.accepts(t) =>
        writer.write("null")
      case t if RangeType.accepts(t) =>
        writeValue(ArrayType.coerce(v))
      case t if ArrayType.accepts(t) =>
        val iterator = v.evaluate.asInstanceOf[ArraySeq].toIterator()
        if (iterator.isEmpty) {
          writer.write("[]")
        } else {
          startArray(v)
          var first = true
          val elements = if (settings.limitCollectionSize()) {
            iterator.take(settings.maxCollectionSize)
          } else {
            iterator
          }
          elements.foreach((v2: Value[_]) => {
            if (!first) {
              writer.write(", ")
              newline()
            }
            writeValue(v2)
            first = false
          })
          endArray(v)
        }
      case t if ObjectType.accepts(t) => {
        val valuePairs = v.evaluate.asInstanceOf[ObjectSeq].toSeq()
        if (valuePairs.isEmpty) {
          writer.write("{}")
        } else {
          val missingNamespace: Seq[Namespace] = collectUndeclaredNS(valuePairs)
          val needNSDeclarations = missingNamespace.nonEmpty
          if (needNSDeclarations) {
            declaredNs.push(mutable.ArrayBuffer())
            writer.write("do ")
            startObject()
            missingNamespace.foreach((namespace) => {
              writeNamespace(getUniquePrefix(namespace), namespace.uri)
            })
            writer.write("---")
            newline()
          }
          startObject()
          var first = true
          val elements = if (settings.limitCollectionSize()) {
            valuePairs.take(settings.maxCollectionSize)
          } else {
            valuePairs
          }
          elements
            .foreach((ekv) => {
              if (!first) {
                writer.write(",")
                newline()
              }
              qualifiedName(ekv._1, AttributeHelper.attributes(ekv._1))
              writer.write(": ")
              writeValue(ekv._2)
              first = false
            })

          if (needNSDeclarations) {
            endObject()
            declaredNs.pop()
          }
          endObject()
        }
      }
      case t if FunctionType.accepts(t) =>
        val parameters = FunctionType.coerce(v).parameters
        writer.write("(")
        var first = true
        parameters.foreach((param) => {
          if (!first) {
            writer.append(", ")
          }
          writer.write(param.name)
          writer.write(":")
          writer.write(param.wtype.name)
          first = false
        })
        writer.write(") -> ???")
      case t if isDateValue(t) => {
        writer.write("|")
        writer.write(StringType.coerce(v).evaluate.toString)
        writer.write("|")
      }
      case t if RegexType.accepts(t) => {
        writer.write("/")
        var regex = RegexType.coerce(v).evaluate.regex
        //Escape the / with \/
        regex = regex.replaceAll("\\/", "\\\\/")
        writer.write(regex)
        writer.write("/")
      }
      case ns if NamespaceType.accepts(ns) => {
        writeValue(ObjectType.coerce(v))
        writer.write("as Namespace")
      }
      case ns if BinaryType.accepts(ns) => {
        val base64String = BinaryHelper.toBase64String(BinaryType.coerce(v).evaluate, ctx.serviceManager.memoryService)
        writeQuoteString(base64String)
        val newSchema = Schema(Seq(SchemaProperty(StringValue(Schema.BASE_PROPERTY_NAME), StringValue("64"))))
        writeSchema(BinaryType, Some(newSchema))
      }
      case ns if StringType.accepts(ns) => {
        val strValue: String = StringType.coerce(v).evaluate.toString
        writeQuoteString(strValue)
      }
      case _ => {
        val strValue: String = StringType.coerce(v).evaluate.toString
        writeQuoteString(strValue)
        //We have just coerced so we shouldn't keep the type. We loose precision :(
        ignoreSchema = true
      }
    }
    if (!ignoreSchema) {
      writeSchema(v.valueType, v.schema)
    }
  }

  private def getUniquePrefix(namespace: Namespace): String = {
    getPrefixFor(namespace.prefix, namespace.uri)
  }

  private def isDateValue(v: Value[_])(implicit ctx: EvaluationContext) = {
    DateTimeType.accepts(v) ||
      LocalDateType.accepts(v) ||
      PeriodType.accepts(v) ||
      LocalDateTimeType.accepts(v) ||
      TimeType.accepts(v) ||
      LocalTimeType.accepts(v) ||
      TimeZoneType.accepts(v)
  }

  private def collectUndeclaredNS(valuePairs: Seq[KeyValuePair])(implicit ctx: EvaluationContext): Seq[Namespace] = {
    valuePairs.flatMap((kvp) => {
      val key = kvp._1
      val keyName = key.evaluate
      val mayBeAttributes = AttributeHelper.attributes(key)
      val missingAttributesNS: TraversableOnce[Namespace] = mayBeAttributes match {
        case Some(attributes) => {
          attributes.evaluate.toIterator().flatMap((nvp) => {
            nvp._1.evaluate.namespace.flatMap((qname) => {
              if (alreadyDefined(qname)) {
                None
              } else {
                Some(qname)
              }
            })
          })
        }
        case None => Seq()
      }

      val missingNS = keyName.namespace.flatMap((qname) => {
        if (alreadyDefined(qname)) {
          None
        } else {
          Some(qname)
        }
      })
      if (missingNS.isDefined)
        missingAttributesNS.toSeq.:+(missingNS.get)
      else
        missingAttributesNS
    }).filter(_.uri.nonEmpty).distinct
  }

  private def writeQuoteString(strValue: String): Unit = {
    writer.write(StringEscapeHelper.escapeString(strValue))
  }

  override def flush(): Unit = writer.flush()

  override def dataFormat: Option[DataFormat[_, _]] = Some(new WeaveDataFormat)
}

object WeaveWriter {
  def apply(tp: TargetProvider, settings: WeaveWriterSettings)(implicit ctx: EvaluationContext): WeaveWriter = {
    new WeaveWriter(tp.asOutputStream, settings)
  }

  def apply(os: OutputStream, settings: WeaveWriterSettings)(implicit ctx: EvaluationContext): WeaveWriter = {
    new WeaveWriter(os, settings)
  }
}

class WeaveWriterSettings extends ConfigurableDeferred with OnlyDataSettings {
  var indent: String = _
  var ignoreSchema: Boolean = _
  var maxCollectionSize: Int = -1

  def limitCollectionSize(): Boolean = maxCollectionSize >= 0

  override def defaultOnlyDataValue: Boolean = true

  override def loadSettingsOptions(): Map[String, ModuleOption] = {
    super.loadSettingsOptions +
      StringModuleOption("indent", defaultValue = "  ", descriptionUrl = "data-format/dw/indent.asciidoc") +
      BooleanModuleOption("ignoreSchema", defaultValue = false, descriptionUrl = "data-format/dw/ignoreSchema.asciidoc") +
      IntModuleOption("maxCollectionSize", defaultValue = -1, descriptionUrl = "data-format/dw/maxCollectionSize.asciidoc")
  }

  protected override def writeSettingsValue(settingName: String, value: Any): Unit = {
    settingName match {
      case "indent" => {
        indent = value.toString
      }
      case "ignoreSchema" => {
        ignoreSchema = value.asInstanceOf[Boolean]
      }
      case "maxCollectionSize" => {
        maxCollectionSize = value.asInstanceOf[Int]
      }
      case _ =>
    }
  }
}
