package org.mule.weave.v2.module.commons.java.writer

import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.structure.QualifiedName
import org.mule.weave.v2.model.structure.schema.Schema
import org.mule.weave.v2.model.values.Value
import org.mule.weave.v2.module.commons.java.JavaClassLoaderHelper
import org.mule.weave.v2.module.commons.java.JavaTypesHelper.reg
import org.mule.weave.v2.module.commons.java.value.JavaValue
import org.mule.weave.v2.module.commons.java.writer.converter.BaseJavaDataConverter
import org.mule.weave.v2.module.commons.java.writer.entry.PropertyAwareEntry
import org.mule.weave.v2.module.commons.java.writer.entry.WriterEntry
import org.mule.weave.v2.module.commons.java.writer.exception.CanNotConvertValueException
import org.mule.weave.v2.module.writer.WriterWithAttributes
import org.mule.weave.v2.parser.location.LocationCapable

import java.util
import java.util.Optional
import scala.collection.mutable

trait BaseJavaWriter extends WriterWithAttributes {

  implicit val converter: BaseJavaDataConverter

  implicit val classLoaderHelper: JavaClassLoaderHelper

  val loader: Option[ClassLoader]

  val entry: mutable.Stack[WriterEntry] = new mutable.Stack[WriterEntry]

  var root: WriterEntry = _

  override def result: Any = if (root != null) JavaAdapter.fromScalaToJava(root.resolveEntryValue()) else null

  override def close(): Unit = {}

  override def doEndDocument(location: LocationCapable): Unit = {}

  override def flush(): Unit = {}

  def write(item: WriterEntry)(implicit ctx: EvaluationContext) {
    if (entry.isEmpty && root == null) {
      root = item
    } else if (entry.nonEmpty) {
      entry.top.putValue(item.resolveEntryValue(), item.schemaOption)
    }
  }

  protected def startArray(location: LocationCapable, schema: Option[Schema], clazzWithRestriction: ClassTypeWithRestriction)(implicit ctx: EvaluationContext): WriterEntry

  protected def endArray(location: LocationCapable)(implicit ctx: EvaluationContext): Unit

  protected def startObject(location: Value[_], schema: Option[Schema], clazzWithRestriction: ClassTypeWithRestriction)(implicit ctx: EvaluationContext): WriterEntry

  protected def endObject(location: LocationCapable)(implicit ctx: EvaluationContext): Unit

  protected def topEntryTypeOption: Option[Class[_]] = {
    if (entry.nonEmpty) {
      Some(entry.top.entryType())
    } else {
      None
    }
  }

  protected def topContentTypeSchemaOption: Option[ClassSchemaNode] = {
    if (entry.nonEmpty) {
      entry.top.contentTypeSchema()
    } else {
      None
    }
  }

  protected def calculateClass(mayBeSchema: Option[Schema], locationCapable: LocationCapable)(implicit ctx: EvaluationContext): ClassTypeWithRestriction = {
    val parentConstrainClass: Option[Class[_]] = topEntryTypeOption
    val parentConstrainClassSchema: Option[ClassSchemaNode] = topContentTypeSchemaOption
    val currentClass = mayBeSchema match {
      case Some(schema) => {
        val className = schema.`class`
        if (className.isDefined) {
          val currentValueClassTree = classLoaderHelper.buildClassSchemaTree(reg.matcher(className.get).replaceAll(""), locationCapable)
          val currentValueClass = classLoaderHelper.loadClass(currentValueClassTree.className, loader, locationCapable)
          if (parentConstrainClass.isDefined) {
            if (parentConstrainClass.get.isAssignableFrom(currentValueClass)) {
              ClassTypeWithRestriction(Some(currentValueClass), Some(currentValueClassTree))
            } else {
              ClassTypeWithRestriction(parentConstrainClass, Some(currentValueClassTree))
            }
          } else {
            ClassTypeWithRestriction(Some(currentValueClass), Some(currentValueClassTree))
          }
        } else {
          ClassTypeWithRestriction(parentConstrainClass, parentConstrainClassSchema)
        }
      }
      case _ => ClassTypeWithRestriction(parentConstrainClass, parentConstrainClassSchema)
    }

    ClassTypeWithRestriction(
      currentClass.classValue
        .map((entryType) => {
          if (entryType.equals(classOf[util.List[_]])) {
            classOf[util.ArrayList[Any]]
          } else if (entryType.equals(classOf[util.Set[_]])) {
            classOf[util.LinkedHashSet[Any]]
          } else if (entryType.equals(classOf[util.Collection[_]])) {
            classOf[util.ArrayList[Any]]
          } else if (entryType.equals(classOf[util.Map[_, _]])) {
            classOf[util.LinkedHashMap[String, Any]]
          } else if (entryType.equals(classOf[Optional[_]])) {
            if (entry.nonEmpty) entry.top.genericType(0).get else entryType
          } else {
            entryType
          }
        }),
      currentClass.constrainClassSchema)
  }

  protected def toJavaValue(value: Any, mayBeSchema: Option[Schema], clazz: Class[_], className: => String, location: LocationCapable)(implicit ctx: EvaluationContext): Any = {
    val to: Option[_] = converter.to(value, mayBeSchema, clazz)
    to match {
      case Some(converted) => converted
      case _               => throw new CanNotConvertValueException(location.location(), value, className)
    }
  }

  protected def isAssignableToRequiredParentType(jv: JavaValue[_])(implicit ctx: EvaluationContext): Boolean = {
    if (jv.underlying() != null) {
      topEntryTypeOption.forall((clazz) => {
        val value = jv.underlying()
        clazz.isAssignableFrom(value.getClass)
      })
    } else {
      //Null is assignable to all the values in java
      true
    }
  }

  protected def key(location: LocationCapable, qname: QualifiedName, schema: Option[Schema]): Unit = {
    entry.top match {
      case pae: PropertyAwareEntry => entry.push(pae.createPropertyEntry(location, qname, schema))
      case t                       => throw new RuntimeException("Unexpected writer entry " + t)
    }
  }
}
