package org.mule.weave.v2.module.dwb.writer

import java.io.ByteArrayInputStream
import java.io.ByteArrayOutputStream
import java.io.FileInputStream
import java.io.FileOutputStream
import java.io.InputStream
import java.io.OutputStream
import java.math.BigInteger
import java.time.LocalDate
import java.time.LocalDateTime
import java.time.LocalTime
import java.time.OffsetTime
import java.time.Period
import java.time.ZoneId
import java.time.ZonedDateTime
import java.util.{ Stack => JStack }
import javax.xml.namespace.QName

import org.mule.weave.v2.dwb.api.WeaveStreamWriter
import org.mule.weave.v2.model.EvaluationContext
import org.mule.weave.v2.model.capabilities.UnknownLocationCapable
import org.mule.weave.v2.model.structure.Namespace
import org.mule.weave.v2.model.structure.QualifiedName
import org.mule.weave.v2.module.dwb.DefaultWeaveBinaryDataFormat
import org.mule.weave.v2.module.dwb.reader.exceptions.DWBRuntimeExecutionException
import org.mule.weave.v2.module.reader.AutoPersistedOutputStream
import org.mule.weave.v2.module.reader.SourceProvider

import scala.collection.mutable
import scala.collection.mutable.ArrayBuffer
import scala.collection.mutable.ListBuffer

/**
  * This class writes in a deferred way because schema properties/attributes presence is indicated
  * in the value token. In array/objects it's in the end token.
  * We need to defer writes until we can determine the current value's not going to have a schema property/attribute,
  * which is when the next key or value is written
  */
class DefaultWeaveStreamWriter(os: OutputStream) extends WeaveStreamWriter {

  private implicit val ctx: EvaluationContext = EvaluationContext()

  private val writer = new WeaveBinaryWriter(os)
  private val putInLC = new JStack[Boolean]()
  putInLC.push(true)

  private val pending = new ListBuffer[WriteCommand]()
  private var strategy: Strategy = new DefaultStrategy()

  private var mode: WriterMode = new DefaultMode()

  private def shouldPutInLC(): Boolean = {
    putInLC.peek()
  }

  override def close(): Unit = {
    writer.close()
  }

  override def flush(): Unit = {
    writer.flush()
  }

  override def getResult: InputStream = {
    os.flush()
    os match {
      case aos: AutoPersistedOutputStream => aos.toInputStream
      case bos: ByteArrayOutputStream     => new ByteArrayInputStream(bos.toByteArray)
      case fos: FileOutputStream          => new FileInputStream(fos.getFD)
      case _                              => throw new RuntimeException("Don't know how to handle this OutputStream: " + os.getClass.getSimpleName)
    }
  }

  /**
    * Write document headers
    */
  override def writeStartDocument(): WeaveStreamWriter = {
    writer.startDocument(UnknownLocationCapable)
    this
  }

  /**
    * Write index (in indexed mode)
    */
  override def writeEndDocument(): WeaveStreamWriter = {
    writePending()
    writer.endDocument(UnknownLocationCapable)
    this
  }

  /**
    * Writes all the pending commands
    */
  private def writePending(): Unit = {
    while (pending.nonEmpty) {
      val command = pending.remove(0)
      command.execute(writer)
    }
  }

  override def writeStartObject(): WeaveStreamWriter = {
    writeValue(new StartObjectWriteCommand())
    this
  }

  override def writeEndObject(): WeaveStreamWriter = {
    writeValue(new EndObjectWriteCommand())
    this
  }

  override def writeStartArray(): WeaveStreamWriter = {
    writeValue(new StartArrayWriteCommand())
    this
  }

  override def writeEndArray(): WeaveStreamWriter = {
    writePending()
    putInLC.pop()
    writeValue(new EndArrayWriteCommand())
    this
  }

  override def writeKey(localName: String): WeaveStreamWriter = {
    writeKey(localName, None)
    this
  }

  override def writeKey(qName: QName): WeaveStreamWriter = {
    val ns = Namespace(qName.getPrefix, qName.getNamespaceURI)
    writeKey(qName.getLocalPart, Some(ns))
    this
  }

  protected def writeKey(name: String, namespace: Option[Namespace]): Unit = {
    mode.doWriteKey(name, namespace, strategy)
  }

  private def writeValue(writeCommand: WriteCommand): Unit = {
    mode.doWriteValue(writeCommand, strategy)
  }

  override def writeInt(number: Int): WeaveStreamWriter = {
    writeValue(new IntWriteCommand(number))
    this
  }

  override def writeLong(number: Long): WeaveStreamWriter = {
    writeValue(new LongWriteCommand(number))
    this
  }

  override def writeBigInt(number: BigInteger): WeaveStreamWriter = {
    writeValue(new BigIntWriteCommand(number))
    this
  }

  override def writeDouble(number: Double): WeaveStreamWriter = {
    writeValue(new DoubleWriteCommand(number))
    this
  }

  override def writeBigDecimal(number: java.math.BigDecimal): WeaveStreamWriter = {
    writeValue(new BigDecimalWriteCommand(number))
    this
  }

  override def writeString(str: String): WeaveStreamWriter = {
    writeValue(new StringWriteCommand(str))
    this
  }

  override def writeDateTime(zonedDateTime: ZonedDateTime): WeaveStreamWriter = {
    writeValue(new DateTimeWriteCommand(zonedDateTime))
    this
  }

  override def writeLocalDate(localDate: LocalDate): WeaveStreamWriter = {
    writeValue(new LocalDateWriteCommand(localDate))
    this
  }

  override def writePeriod(period: Period): WeaveStreamWriter = {
    throw new DWBRuntimeExecutionException("Not implemented yet")
  }

  override def writeLocalDateTime(localDateTime: LocalDateTime): WeaveStreamWriter = {
    writeValue(new LocalDateTimeWriteCommand(localDateTime))
    this
  }

  override def writeTime(offsetTime: OffsetTime): WeaveStreamWriter = {
    writeValue(new OffsetTimeWriteCommand(offsetTime))
    this
  }

  override def writeLocalTime(localTime: LocalTime): WeaveStreamWriter = {
    writeValue(new LocalTimeWriteCommand(localTime))
    this
  }

  override def writeTimeZone(zoneId: ZoneId): WeaveStreamWriter = {
    writeValue(new TimeZoneWriteCommand(zoneId))
    this
  }

  override def writeBinary(bytes: Array[Byte]): WeaveStreamWriter = {
    writeValue(new ByteArrayWriteCommand(bytes))
    this
  }

  override def writeBinary(inputStream: InputStream): WeaveStreamWriter = {
    writeValue(new InputStreamWriteCommand(inputStream))
    this
  }

  override def writeRegex(regex: String): WeaveStreamWriter = {
    throw new DWBRuntimeExecutionException("Not implemented yet")
  }

  override def writeRange(from: Int, to: Int): WeaveStreamWriter = {
    throw new DWBRuntimeExecutionException("Not implemented yet")
  }

  override def writeNull(): WeaveStreamWriter = {
    writeValue(new NullWriteCommand())
    this
  }

  override def beginTransaction(): WeaveStreamWriter = {
    writePending()
    strategy = new TransactionStrategy()
    this
  }

  override def endTransaction(): WeaveStreamWriter = {
    strategy.commit()
    strategy = new DefaultStrategy()
    this
  }

  override def commit(): WeaveStreamWriter = {
    strategy.commit()
    this
  }

  override def rollback(): WeaveStreamWriter = {
    strategy.rollback()
    this
  }

  override def mergeStream(dwb: InputStream): WeaveStreamWriter = {
    writePending()
    val reader = DefaultWeaveBinaryDataFormat.reader(SourceProvider(dwb))
    val dwbToMerge = reader.read("dwbToMerge")
    writer.doWriteValue0(dwbToMerge, shouldPutInLC())
    this
  }

  override def writeStartAttribute(): WeaveStreamWriter = {
    mode = new AttributeMode()
    putInLC.push(false)
    this
  }

  override def writeEndAttribute(): WeaveStreamWriter = {
    mode = new DefaultMode()
    writePending()
    putInLC.pop
    this
  }

  def writeStartSchema(): WeaveStreamWriter = {
    mode = new SchemaMode()
    putInLC.push(false)
    this
  }

  def writeEndSchema(): WeaveStreamWriter = {
    mode = new DefaultMode()
    writePending()
    putInLC.pop
    this
  }

  trait WriterMode {
    def doWriteKey(name: String, namespace: Option[Namespace], strategy: Strategy): Unit

    def doWriteValue(writeCommand: WriteCommand, strategy: Strategy): Unit
  }

  class DefaultMode extends WriterMode {

    override def doWriteKey(name: String, namespace: Option[Namespace], strategy: Strategy): Unit = {
      strategy.handlePending()
      val qName = QualifiedName(name, namespace)
      pending += new KeyWriteCommand(qName)
    }

    override def doWriteValue(writeCommand: WriteCommand, strategy: Strategy): Unit = {
      strategy.handlePending()
      pending += writeCommand
    }
  }

  class SchemaMode extends WriterMode {
    override def doWriteKey(name: String, namespace: Option[Namespace], strategy: Strategy): Unit = {
      pending.last.addPropertyKey(name)
    }

    override def doWriteValue(writeCommand: WriteCommand, strategy: Strategy): Unit = {
      pending.last.addPropertyValue(writeCommand)
    }
  }

  class AttributeMode extends WriterMode {
    override def doWriteKey(name: String, namespace: Option[Namespace], strategy: Strategy): Unit = {
      val qName = QualifiedName(name, namespace)
      pending.last.addAttributeKey(qName)
    }

    override def doWriteValue(writeCommand: WriteCommand, strategy: Strategy): Unit = {
      pending.last.addAttributeValue(writeCommand)
    }
  }

  trait Strategy {
    def handlePending(): Unit
    def commit(): Unit
    def rollback(): Unit
  }

  /**
    * This strategy doesn't flush the pending commands until a commit is made or the transaction is over
    */
  class TransactionStrategy extends Strategy {
    def handlePending(): Unit = {
      //don't write pending commands
    }

    override def commit(): Unit = {
      //FIXME if they commit the previous value can no longer have attributes/schema_props
      writePending()
    }

    override def rollback(): Unit = {
      pending.clear()
    }
  }

  /**
    * This strategy flushes the pending commands before adding to the pending list the new command
    */
  class DefaultStrategy extends Strategy {
    def handlePending(): Unit = {
      writePending()
    }
    override def commit(): Unit = {
      // Do nothing, since it's not a transaction
    }
    override def rollback(): Unit = {
      // Do nothing, since it's not a transaction
    }
  }

  /**
    * This object is used to store the properties until the value is written
    */
  trait WriteCommand {
    private val propKeys = new ArrayBuffer[String](0)
    private val propValues = new ArrayBuffer[WriteCommand](0)
    val attrKeys = new ArrayBuffer[QualifiedName](0)
    val attrValues = new ArrayBuffer[WriteCommand](0)

    def addPropertyKey(name: String): Unit = {
      propKeys += name
    }

    def addPropertyValue(value: WriteCommand): Unit = {
      propValues += value
    }

    def addAttributeKey(qname: QualifiedName): Unit = {
      attrKeys += qname
    }

    def addAttributeValue(value: WriteCommand): Unit = {
      attrValues += value
    }

    def getPropsMap(): mutable.Map[String, WriteCommand] = {
      val propSeq = propKeys.zip(propValues)
      mutable.LinkedHashMap(propSeq: _*)
    }

    final def execute(writer: WeaveBinaryWriter): Unit = {
      val propsMap = getPropsMap()
      val hasSchema = propsMap.nonEmpty
      doWrite(writer, hasSchema)
      if (hasSchema) {
        //schemas are written after the value
        writer.writeShort(propsMap.size)
        for ((key, value) <- propsMap) {
          writer.writeSchemaPropertyKey(key)
          value.execute(writer)
        }
      }
    }

    protected def doWrite(writer: WeaveBinaryWriter, hasSchema: Boolean): Unit

  }

  class KeyWriteCommand(qName: QualifiedName) extends WriteCommand {

    override def doWrite(writer: WeaveBinaryWriter, hasSchema: Boolean): Unit = {
      import WeaveBinaryWriter.getKeyTokenType
      val hasAttrs = attrKeys.nonEmpty
      val keyTokenType = getKeyTokenType(qName.namespace.isDefined, hasAttrs)
      writer.writeQName(qName, keyTokenType, putTokenInLC = true)
      if (hasAttrs) {
        //attributes are written after the key
        writer.writeShort(attrKeys.size)
        for ((key, value) <- attrKeys.zip(attrValues)) {
          writer.writeAttributeKey(key)
          value.execute(writer)
        }
      }
    }

  }

  /**
    * This commands remove the class since that is already specified in the type of command.
    */
  trait NumberWriteCommand extends WriteCommand {
    override def getPropsMap(): mutable.Map[String, WriteCommand] = {
      val propsMap = super.getPropsMap()
      propsMap.remove("class")
      propsMap
    }

  }

  class IntWriteCommand(number: Int) extends NumberWriteCommand {
    override def doWrite(writer: WeaveBinaryWriter, hasSchema: Boolean): Unit = {
      writer.writeInt(shouldPutInLC(), hasSchema, number)
    }
  }

  class LongWriteCommand(number: Long) extends NumberWriteCommand {
    override def doWrite(writer: WeaveBinaryWriter, hasSchema: Boolean): Unit = {
      writer.writeLong(shouldPutInLC(), hasSchema, number)
    }
  }

  class BigIntWriteCommand(number: BigInteger) extends NumberWriteCommand {
    override def doWrite(writer: WeaveBinaryWriter, hasSchema: Boolean): Unit = {
      writer.writeBigInt(shouldPutInLC(), hasSchema, number)
    }
  }

  class DoubleWriteCommand(number: Double) extends NumberWriteCommand {
    override def doWrite(writer: WeaveBinaryWriter, hasSchema: Boolean): Unit = {
      writer.writeDouble(shouldPutInLC(), hasSchema, number)
    }
  }

  class BigDecimalWriteCommand(number: java.math.BigDecimal) extends NumberWriteCommand {
    override def doWrite(writer: WeaveBinaryWriter, hasSchema: Boolean): Unit = {
      writer.writeBigDecimal(shouldPutInLC(), hasSchema, number)
    }
  }

  class StringWriteCommand(str: String) extends WriteCommand {
    override def doWrite(writer: WeaveBinaryWriter, hasSchema: Boolean): Unit = {
      writer.writeString(shouldPutInLC(), hasSchema, str)
    }
  }

  class DateTimeWriteCommand(dateTime: ZonedDateTime) extends WriteCommand {
    override def doWrite(writer: WeaveBinaryWriter, hasSchema: Boolean): Unit = {
      writer.writeDateTime(shouldPutInLC(), hasSchema, dateTime)
    }
  }

  class LocalDateTimeWriteCommand(dateTime: LocalDateTime) extends WriteCommand {
    override def doWrite(writer: WeaveBinaryWriter, hasSchema: Boolean): Unit = {
      writer.writeLocalDateTime(shouldPutInLC(), hasSchema, dateTime)
    }
  }

  class ByteArrayWriteCommand(binary: Array[Byte]) extends WriteCommand {
    override def doWrite(writer: WeaveBinaryWriter, hasSchema: Boolean): Unit = {
      writer.writeBinary(shouldPutInLC(), hasSchema, binary)
    }
  }

  class InputStreamWriteCommand(binary: InputStream)(implicit ctx: EvaluationContext) extends WriteCommand {
    override def doWrite(writer: WeaveBinaryWriter, hasSchema: Boolean): Unit = {
      writer.writeBinary(shouldPutInLC(), hasSchema, binary)
    }
  }

  class OffsetTimeWriteCommand(time: OffsetTime) extends WriteCommand {
    override def doWrite(writer: WeaveBinaryWriter, hasSchema: Boolean): Unit = {
      writer.writeOffsetTime(shouldPutInLC(), hasSchema, time)
    }
  }

  class LocalDateWriteCommand(localDate: LocalDate) extends WriteCommand {
    override def doWrite(writer: WeaveBinaryWriter, hasSchema: Boolean): Unit = {
      writer.writeLocalDate(shouldPutInLC(), hasSchema, localDate)
    }
  }

  class LocalTimeWriteCommand(localTime: LocalTime) extends WriteCommand {
    override def doWrite(writer: WeaveBinaryWriter, hasSchema: Boolean): Unit = {
      writer.writeLocalTime(shouldPutInLC(), hasSchema, localTime)
    }
  }

  class TimeZoneWriteCommand(zoneId: ZoneId) extends WriteCommand {
    override def doWrite(writer: WeaveBinaryWriter, hasSchema: Boolean): Unit = {
      writer.writeTimeZone(shouldPutInLC(), hasSchema, zoneId)
    }
  }

  class NullWriteCommand() extends WriteCommand {
    override def doWrite(writer: WeaveBinaryWriter, hasSchema: Boolean): Unit = {
      writer.writeNull(shouldPutInLC(), hasSchema)
    }
  }

  class EndArrayWriteCommand() extends WriteCommand {
    override def doWrite(writer: WeaveBinaryWriter, hasSchema: Boolean): Unit = {
      writer.writeEndArray(hasSchema)
    }
  }

  class EndObjectWriteCommand() extends WriteCommand {
    override def doWrite(writer: WeaveBinaryWriter, hasSchema: Boolean): Unit = {
      writer.writeEndObject(hasSchema)
      putInLC.pop()
    }
  }

  class StartArrayWriteCommand() extends WriteCommand {
    override def doWrite(writer: WeaveBinaryWriter, hasSchema: Boolean): Unit = {
      writer.writeStartArray(shouldPutInLC())
      putInLC.push(true)
    }
  }

  class StartObjectWriteCommand() extends WriteCommand {
    override def doWrite(writer: WeaveBinaryWriter, hasSchema: Boolean): Unit = {
      writer.writeStartObject(shouldPutInLC())
      putInLC.push(false)
    }
  }
}
