package org.mule.weave.v2.module.dwb.reader.indexed

import java.io.DataInputStream
import java.nio.charset.Charset

import org.mule.weave.v2.model.structure.NameSeq
import org.mule.weave.v2.model.structure.NameValuePair
import org.mule.weave.v2.model.structure.Namespace
import org.mule.weave.v2.model.structure.QualifiedName
import org.mule.weave.v2.model.values.AttributesValue
import org.mule.weave.v2.model.values.KeyValue
import org.mule.weave.v2.model.values.Value
import org.mule.weave.v2.module.dwb.DwTokenHelper
import org.mule.weave.v2.module.dwb.DwTokenType

import scala.collection.mutable.ArrayBuffer

object BinaryValueRetriever {
  private val charset: Charset = Charset.forName("UTF-8")

  /**
    * First reads the length which is specified in a short, then the string
    */
  def readShortString(input: DataInputStream): String = {
    val length = input.readUnsignedShort()
    readString(input, length)
  }

  def readShortString(input: DataInputStream, buffer: Array[Byte]): String = {
    val length = input.readUnsignedShort()
    readString(input, length, buffer)
  }

  def readString(input: DataInputStream, length: Int): String = {
    val buffer = new Array[Byte](length)
    readString(input, length, buffer)
  }

  def readString(input: DataInputStream, length: Int, buffer: Array[Byte]): String = {
    input.readFully(buffer, 0, length)
    new String(buffer, 0, length, charset)
  }

  def readKey(keyTokenIndex: Long, keyToken: Array[Long], input: BinaryParserInput): KeyValue = {
    val name = WeaveBinaryValue.readLocalName(input, keyToken)
    val tokenType = DwTokenHelper.getTokenType(keyToken)

    tokenType match {
      case DwTokenType.Key =>
        val qName = QualifiedName(name, None)
        KeyValue(qName, None)

      case DwTokenType.KeyWithNS =>
        val namespace = readNamespace(keyToken, input)
        val qName = QualifiedName(name, Some(namespace))
        KeyValue(qName, None)

      case DwTokenType.KeyWithAttr =>
        val qName = QualifiedName(name, None)
        val attrs = readAttributes(keyTokenIndex, input)
        KeyValue(qName, Some(attrs))

      case DwTokenType.KeyWithNSAttr =>
        val namespace = readNamespace(keyToken, input)
        val qName = QualifiedName(name, Some(namespace))
        val attrs = readAttributes(keyTokenIndex, input)
        KeyValue(qName, Some(attrs))

      case _ =>
        throw new IllegalArgumentException("Got '" + DwTokenType.getNameFor(tokenType) + "' token while reading keys")
    }
  }

  /**
    * Reads an index from the stream and returns the Namespace it references.
    * Note: this assumes the Namespace was previously declared.
    */
  def readNamespace(token: Array[Long], input: BinaryParserInput): Namespace = {
    val nsIndexMaybe = DwTokenHelper.getNsIndex(token)
    val namespace = input.namespaces(nsIndexMaybe.get)
    namespace
  }

  def readAttributes(keyTokenIndex: Long, input: BinaryParserInput): Value[NameSeq] = {
    val tokenArray = input.tokenArray
    val keyToken = tokenArray(keyTokenIndex)
    val offset = DwTokenHelper.getOffset(keyToken)
    input.seekableStream.seek(offset)
    var attrPairs = new ArrayBuffer[NameValuePair]()
    val dis = input.dataInputStream
    val attrCount = dis.readUnsignedShort()

    var i = 0
    while (i < attrCount) {
      val pairPos = keyTokenIndex + (i * 2)
      val nameStr = WeaveBinaryValue.readLocalName(input, tokenArray(pairPos + 1))
      val key = KeyValue(nameStr)
      val value = WeaveBinaryValue.apply(pairPos + 2, None, input)
      attrPairs += NameValuePair(key, value)
      i += 1
    }
    AttributesValue(attrPairs)
  }
}
