package org.mule.weave.v2.debugger

/**
  * Represents a value in the runtime
  */
sealed trait DebuggerValue extends Serializable {
  /**
    * Returns the type name of t
    * @return The name
    */
  def typeName(): String

  /**
    * The location of value
    * @return The location
    */
  def location(): DebuggerLocation

}

@SerialVersionUID(1000L)
case class ObjectDebuggerValue(fields: Array[FieldDebuggerValue], typeName: String, location: DebuggerLocation) extends DebuggerValue {
  override def toString = s"{${fields.map(_.toString).mkString(", ")}}"
}

@SerialVersionUID(1000L)
case class FieldDebuggerValue(key: KeyDebuggerValue, value: DebuggerValue, location: DebuggerLocation) extends DebuggerValue {
  override def toString = s"'$key': $value"

  override def typeName(): String = "KeyValue Pair"
}

@SerialVersionUID(1000L)
case class AttributeDebuggerValue(name: String, value: DebuggerValue, location: DebuggerLocation) extends DebuggerValue {
  override def toString = s"'$name' : $value"

  override def typeName(): String = "Attribute"
}

@SerialVersionUID(1000L)
case class KeyDebuggerValue(name: String, attr: Array[AttributeDebuggerValue], location: DebuggerLocation) extends DebuggerValue {
  override def toString = {
    val attrsToString = if (attr.nonEmpty) s" @(${attr.mkString(", ")})" else ""
    s"$name$attrsToString"
  }

  override def typeName(): String = "Key"
}

@SerialVersionUID(1000L)
case class ArrayDebuggerValue(values: Array[DebuggerValue], typeName: String, location: DebuggerLocation) extends DebuggerValue {
  override def toString = s"[ ${values.map(_.toString).mkString(", ")} ]"
}

@SerialVersionUID(1000L)
case class DebuggerFunction(parameters: Array[String], typeName: String, location: DebuggerLocation) extends DebuggerValue {
  override def toString = s"function(${parameters.mkString(", ")})"
}

@SerialVersionUID(1000L)
case class SimpleDebuggerValue(value: String, typeName: String, location: DebuggerLocation) extends DebuggerValue {
  override def toString = s"$value"
}
