/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal;

import static java.util.Collections.emptyList;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static java.util.Optional.ofNullable;
import static java.util.stream.Collectors.toList;
import static java.util.stream.IntStream.range;
import static org.mule.runtime.module.extension.api.loader.java.type.PropertyElement.Accessibility.READ_ONLY;
import static org.mule.runtime.module.extension.api.loader.java.type.PropertyElement.Accessibility.READ_WRITE;
import static org.mule.runtime.module.extension.api.loader.java.type.PropertyElement.Accessibility.WRITE_ONLY;
import static org.mule.runtime.module.extension.internal.util.IntrospectionUtils.getApiMethods;
import static org.mule.runtime.module.extension.internal.util.IntrospectionUtils.getMethodsStream;

import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.ast.api.ASTTypeLoader;
import org.mule.metadata.ast.api.TypeUtils;
import org.mule.metadata.ast.internal.ClassInformationAnnotationFactory;
import org.mule.metadata.java.api.annotation.ClassInformationAnnotation;
import org.mule.runtime.api.util.LazyValue;
import org.mule.runtime.ast.internal.typevisitor.MuleTypeVisitor;
import org.mule.runtime.ast.internal.typevisitor.TypeIntrospectionResult;
import org.mule.runtime.module.extension.api.loader.java.type.AnnotationValueFetcher;
import org.mule.runtime.module.extension.api.loader.java.type.FieldElement;
import org.mule.runtime.module.extension.api.loader.java.type.MethodElement;
import org.mule.runtime.module.extension.api.loader.java.type.OperationElement;
import org.mule.runtime.module.extension.api.loader.java.type.PropertyElement;
import org.mule.runtime.module.extension.api.loader.java.type.PropertyElement.Accessibility;
import org.mule.runtime.module.extension.api.loader.java.type.Type;
import org.mule.runtime.module.extension.api.loader.java.type.TypeGeneric;
import org.mule.runtime.module.extension.internal.util.IntrospectionUtils;

import java.lang.annotation.Annotation;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.stream.Stream;

import javax.annotation.processing.ProcessingEnvironment;
import javax.lang.model.element.TypeElement;
import javax.lang.model.type.TypeMirror;
import javax.lang.model.type.WildcardType;
import javax.lang.model.util.Types;

import org.apache.commons.lang3.builder.EqualsBuilder;
import org.apache.commons.lang3.builder.HashCodeBuilder;

/**
 * {@link Type} implementation which uses {@link TypeMirror} and {@link TypeElement} to represent a Java Class.
 *
 * @since 1.0
 */
public class ASTType implements Type {

  private final TypeMirror typeMirror;
  private final LazyValue<List<TypeGeneric>> typeGenerics;
  private LazyValue<List<OperationElement>> methods;
  private LazyValue<ClassInformationAnnotation> classInformation;
  final ProcessingEnvironment processingEnvironment;
  final TypeElement typeElement;
  ASTUtils astUtils;
  private ASTTypeLoader typeLoader;

  /**
   * Creates a new {@link ASTType} based on a {@link TypeMirror}.
   * 
   * @param typeMirror
   * @param processingEnvironment
   */
  public ASTType(TypeMirror typeMirror, ProcessingEnvironment processingEnvironment) {

    this.processingEnvironment = processingEnvironment;
    TypeIntrospectionResult accept =
        typeMirror.accept(new MuleTypeVisitor(processingEnvironment), TypeIntrospectionResult.builder());
    this.typeElement = accept.getConcreteType();
    this.typeGenerics = new LazyValue<>(() -> toTypeGenerics(accept, processingEnvironment));
    this.typeMirror = typeMirror;
    init();
  }

  public ASTType(TypeElement typeElement, ProcessingEnvironment processingEnvironment) {
    this.processingEnvironment = processingEnvironment;
    this.typeElement = typeElement;
    this.typeMirror = typeElement.asType();
    this.typeGenerics = new LazyValue<>(emptyList());
    init();
  }

  private void init() {
    typeLoader =
        new ASTTypeLoader(processingEnvironment, Collections.singletonList(new ExtensionTypeHandler(processingEnvironment)),
                          new ExtensionTypeObjectFieldHandler(processingEnvironment));
    astUtils = new ASTUtils(processingEnvironment);
    methods = new LazyValue<>(() -> getApiMethods(typeElement, processingEnvironment)
        .stream()
        .map(elem -> new OperationElementAST(elem, processingEnvironment))
        .collect(toList()));
    classInformation = new LazyValue<>(() -> ClassInformationAnnotationFactory.fromTypeMirror(typeMirror, processingEnvironment));
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Optional<Class<?>> getDeclaringClass() {
    return empty();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getName() {
    return typeElement.getSimpleName().toString();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<FieldElement> getFields() {
    return IntrospectionUtils.getFields(typeElement, processingEnvironment)
        .stream()
        .map(elem -> new FieldTypeElement(elem, processingEnvironment))
        .collect(toList());
  }

  @Override
  public List<PropertyElement> getProperties() {
    return TypeUtils.getProperties(typeElement, processingEnvironment)
        .stream().map(property -> {
          Accessibility accessibility;
          if (property.getGetterMethod() != null && property.getSetterMethod() != null) {
            accessibility = READ_WRITE;
          } else if (property.getGetterMethod() != null && property.getGetterMethod() == null) {
            accessibility = READ_ONLY;
          } else {
            accessibility = WRITE_ONLY;
          }
          return PropertyElement.builder()
              .type(new ASTType(property.getType(), processingEnvironment))
              .name(property.getBeanName())
              .accessibility(accessibility)
              .build();
        })
        .collect(toList());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<FieldElement> getAnnotatedFields(Class<? extends Annotation>... annotations) {
    return getFields().stream()
        .filter(elem -> Stream.of(annotations)
            .anyMatch(annotation -> elem.getAnnotation(annotation).isPresent()))
        .collect(toList());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public <A extends Annotation> Optional<A> getAnnotation(Class<A> annotationClass) {
    return ofNullable(typeElement.getAnnotation(annotationClass));
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public <A extends Annotation> Optional<AnnotationValueFetcher<A>> getValueFromAnnotation(Class<A> annotationClass) {
    if (this.isAnnotatedWith(annotationClass)) {
      return of(astUtils.fromAnnotation(annotationClass, typeElement));
    } else {
      return empty();
    }
  }

  /**
   * {@inheritDoc}
   */
  public Optional<TypeElement> getElement() {
    return ofNullable(typeElement);
  }

  @Override
  public Optional<MethodElement> getMethod(String name, Class<?>... parameterTypes) {
    Optional method = getMethodsStream(typeElement, true, processingEnvironment)
        .map(elem -> new OperationElementAST(elem, processingEnvironment))
        .filter(m -> m.getName().equals(name) && m.getParameters().size() == parameterTypes.length
            && range(0, parameterTypes.length)
                .allMatch(index -> m.getParameters().get(index).getType().isSameType(parameterTypes[index])))
        .findFirst();
    return method;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<TypeGeneric> getGenerics() {
    return typeGenerics.get();
  }

  @Override
  public MetadataType asMetadataType() {
    return typeLoader.load(typeMirror)
        .orElseThrow(() -> new RuntimeException("Unable to obtain the MetadataType for the current type: "
            + typeMirror.toString()));
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isAssignableTo(Class<?> clazz) {
    Types types = processingEnvironment.getTypeUtils();
    return types
        .isAssignable(types.erasure(typeMirror), types.erasure(getTypeMirrorFromClass(clazz)));
  }

  @Override
  public boolean isAssignableTo(Type type) {
    if (type instanceof ASTType) {
      return processingEnvironment.getTypeUtils()
          .isAssignable(processingEnvironment.getTypeUtils().erasure(typeMirror),
                        processingEnvironment.getTypeUtils().erasure(((ASTType) type).typeMirror));
    } else if (type.getDeclaringClass().isPresent()) {
      return processingEnvironment.getTypeUtils()
          .isAssignable(processingEnvironment.getTypeUtils().erasure(typeMirror),
                        getTypeMirrorFromClass(type.getDeclaringClass().get()));
    }
    return false;
  }

  @Override
  public boolean isSameType(Type type) {
    if (type instanceof ASTType) {
      TypeMirror givenType = processingEnvironment.getTypeUtils().erasure(((ASTType) type).typeMirror);
      TypeMirror actualType = processingEnvironment.getTypeUtils().erasure(typeMirror);
      return processingEnvironment.getTypeUtils().isSameType(actualType, givenType);
    } else if (type.getDeclaringClass().isPresent()) {
      return isSameType(type.getDeclaringClass().get());
    }
    return false;
  }

  @Override
  public boolean isSameType(Class<?> clazz) {
    Types types = processingEnvironment.getTypeUtils();
    return types
        .isSameType(types.erasure(typeMirror), types.erasure(getTypeMirrorFromClass(clazz)));
  }

  @Override
  public boolean isInstantiable() {
    return getClassInformation().isInstantiable();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isAssignableFrom(Class<?> clazz) {
    TypeMirror typeMirror = getTypeMirrorFromClass(clazz);
    return processingEnvironment.getTypeUtils()
        .isAssignable(typeMirror, processingEnvironment.getTypeUtils().erasure(this.typeMirror));
  }

  @Override
  public boolean isAssignableFrom(Type type) {
    return type.isAssignableTo(this);
  }

  private TypeMirror getTypeMirrorFromClass(Class<?> clazz) {
    return astUtils.getPrimitiveTypeMirror(clazz)
        .orElseGet(() -> {
          if (clazz.isArray()) {
            Class<?> componentType = clazz.getComponentType();
            return processingEnvironment
                .getTypeUtils()
                .getArrayType(getTypeMirrorFromClass(componentType));
          } else {
            return processingEnvironment.getElementUtils()
                .getTypeElement(clazz.getName())
                .asType();
          }
        });
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getTypeName() {
    return typeElement == null ? typeMirror.toString() : typeElement.toString();
  }

  @Override
  public ClassInformationAnnotation getClassInformation() {
    return classInformation.get();
  }

  @Override
  public boolean isAnyType() {
    return typeMirror instanceof WildcardType
        && ((WildcardType) typeMirror).getSuperBound() == null
        && ((WildcardType) typeMirror).getExtendsBound() == null;
  }

  public List<OperationElement> getMethods() {
    return methods.get();
  }

  private List<TypeGeneric> toTypeGenerics(TypeIntrospectionResult result, ProcessingEnvironment pe) {
    List<TypeGeneric> typeGenerics = new ArrayList<>();
    for (TypeIntrospectionResult aResult : result.getGenerics()) {
      typeGenerics.add(new TypeGeneric(new ASTType(aResult.getConcreteTypeMirror(), pe),
                                       aResult.getGenerics().isEmpty() ? emptyList() : toTypeGenerics(aResult, pe)));
    }
    return typeGenerics;
  }

  /**
   * {@inheritDoc}
   */
  public List<Type> getSuperTypeGenerics(Class superTypeClass) {
    return IntrospectionTypeUtils.getSuperTypeGenerics(typeMirror, superTypeClass, processingEnvironment)
        .stream()
        .map(typeMirror -> new ASTType(typeMirror, processingEnvironment))
        .collect(toList());
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }

    if (o == null || getClass() != o.getClass()) {
      return false;
    }

    ASTType type = (ASTType) o;

    return new EqualsBuilder()
        .append(typeMirror, type.typeMirror)
        .append(typeElement, type.typeElement)
        .isEquals();
  }

  public TypeMirror getTypeMirror() {
    return typeMirror;
  }

  @Override
  public int hashCode() {
    return new HashCodeBuilder(17, 37)
        .append(typeMirror)
        .append(typeElement)
        .toHashCode();
  }
}
