/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.connectivity.internal.platform.schema.generator;

import static org.mule.runtime.api.util.Preconditions.checkState;

import org.mule.metadata.api.model.MetadataType;
import org.mule.runtime.api.meta.model.connection.ConnectionProviderModel;
import org.mule.runtime.api.meta.model.parameter.ParameterModel;
import org.mule.runtime.connectivity.api.platform.schema.extension.ExcludeFromConnectivitySchemaModelProperty;
import org.mule.runtime.connectivity.api.platform.schema.generator.ConnectivitySchemaGenerator;
import org.mule.runtime.connectivity.api.platform.schema.generator.ConnectivitySchemaGeneratorBuilder;

import java.util.Set;
import java.util.function.BiFunction;
import java.util.function.Function;

/**
 * Default implementation of {@link ConnectivitySchemaGeneratorBuilder}
 *
 * @since 1.0
 */
public class DefaultConnectivitySchemaGeneratorBuilder implements ConnectivitySchemaGeneratorBuilder {

  private Function<ConnectionProviderModel, Set<String>> connectionTermsExtractor;
  private Function<ParameterModel, Set<String>> parameterTermsExtractor;
  private Function<MetadataType, Set<String>> typeTermsExtractor;
  private Function<ConnectionProviderModel, Boolean> connectionPredicate;
  private BiFunction<ConnectionProviderModel, ParameterModel, Boolean> parameterPredicate;

  /**
   * {@inheritDoc}
   */
  @Override
  public ConnectivitySchemaGeneratorBuilder setConnectionTermsExtractor(Function<ConnectionProviderModel, Set<String>> connectionTermsExtractor) {
    this.connectionTermsExtractor = connectionTermsExtractor;
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ConnectivitySchemaGeneratorBuilder setParameterTermsExtractor(Function<ParameterModel, Set<String>> parameterTermsExtractor) {
    this.parameterTermsExtractor = parameterTermsExtractor;
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ConnectivitySchemaGeneratorBuilder setTypeTermsExtractor(Function<MetadataType, Set<String>> typeTermsExtractor) {
    this.typeTermsExtractor = typeTermsExtractor;
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ConnectivitySchemaGeneratorBuilder setConnectionPredicate(Function<ConnectionProviderModel, Boolean> predicate) {
    connectionPredicate = predicate;
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ConnectivitySchemaGeneratorBuilder setParameterPredicate(BiFunction<ConnectionProviderModel, ParameterModel, Boolean> predicate) {
    parameterPredicate = predicate;
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ConnectivitySchemaGenerator build() {
    checkState(connectionTermsExtractor != null, "Specifying a connectionTermsExtractor is mandatory");
    checkState(parameterTermsExtractor != null, "Specifying a parameterTermsExtractor is mandatory");
    checkState(typeTermsExtractor != null, "Specifying a typeTermsExtractor is mandatory");

    return new DefaultConnectivitySchemaGenerator(connectionTermsExtractor,
                                                  parameterTermsExtractor,
                                                  typeTermsExtractor,
                                                  getConnectionPredicate(),
                                                  getParameterPredicate());
  }

  private Function<ConnectionProviderModel, Boolean> getConnectionPredicate() {
    return connectionPredicate != null
        ? connectionPredicate
        : model -> !model.getModelProperty(ExcludeFromConnectivitySchemaModelProperty.class).isPresent();
  }

  private BiFunction<ConnectionProviderModel, ParameterModel, Boolean> getParameterPredicate() {
    return parameterPredicate != null
        ? parameterPredicate
        : (c, p) -> !p.getModelProperty(ExcludeFromConnectivitySchemaModelProperty.class).isPresent();
  }
}
