/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.connectivity.api.platform.schema.generator;

import org.mule.api.annotation.NoImplement;
import org.mule.metadata.api.model.MetadataType;
import org.mule.runtime.api.meta.model.connection.ConnectionProviderModel;
import org.mule.runtime.api.meta.model.parameter.ParameterModel;
import org.mule.runtime.connectivity.internal.platform.schema.generator.DefaultConnectivitySchemaGeneratorBuilder;

import java.util.Set;
import java.util.function.BiFunction;
import java.util.function.Function;

/**
 * Builder for creating {@link ConnectivitySchemaGenerator} instances
 *
 * @since 1.0
 */
@NoImplement
public interface ConnectivitySchemaGeneratorBuilder {

  /**
   * @return a new builder instance
   */
  static ConnectivitySchemaGeneratorBuilder newInstance() {
    return new DefaultConnectivitySchemaGeneratorBuilder();
  }

  /**
   * Specifies the {@link Function} which will be used to extract the semantic terms out of a {@link ConnectionProviderModel}
   * instance.
   * <p>
   * Providing a function through this method is mandatory. {@link #build()} method will fail if this method has not been
   * previously invoked
   *
   * @param connectionTermsExtractor a extraction function
   * @return {@code this} builder
   */
  ConnectivitySchemaGeneratorBuilder setConnectionTermsExtractor(Function<ConnectionProviderModel, Set<String>> connectionTermsExtractor);

  /**
   * Specifies the {@link Function} which will be used to extract the semantic terms out of a {@link ParameterModel} instance.
   * <p>
   * Providing a function through this method is mandatory. {@link #build()} method will fail if this method has not been
   * previously invoked
   *
   * @param parameterTermsExtractor a extraction function
   * @return {@code this} builder
   */
  ConnectivitySchemaGeneratorBuilder setParameterTermsExtractor(Function<ParameterModel, Set<String>> parameterTermsExtractor);

  /**
   * Specifies the {@link Function} which will be used to extract the semantic terms out of a {@link MetadataType} instance.
   * <p>
   * Providing a function through this method is mandatory. {@link #build()} method will fail if this method has not been
   * previously invoked
   *
   * @param typeTermsExtractor a extraction function
   * @return {@code this} builder
   */
  ConnectivitySchemaGeneratorBuilder setTypeTermsExtractor(Function<MetadataType, Set<String>> typeTermsExtractor);

  /**
   * Specifies a {@link Function} which evaluates if a {@link ConnectionProviderModel} should be added to the generated schema or
   * not. Models for which this predicate yields a {@code false} value will be ignored.
   *
   * @param predicate A predicate Function
   * @return {@code this} builder
   */
  ConnectivitySchemaGeneratorBuilder setConnectionPredicate(Function<ConnectionProviderModel, Boolean> predicate);

  /**
   * Specifies a {@link Function} which evaluates if a {@link ParameterModel} should be added to the generated schema or not.
   * Models for which this predicate yields a {@code false} value will be ignored.
   *
   * @param predicate A predicate Function
   * @return {@code this} builder
   */
  ConnectivitySchemaGeneratorBuilder setParameterPredicate(BiFunction<ConnectionProviderModel, ParameterModel, Boolean> predicate);

  /**
   * Creates and returns a new generator based on this configuration.
   *
   * @return a new {@link ConnectivitySchemaGenerator}
   * @throws IllegalStateException if any of the mandatory methods haven't been previously invoked on {@code this} instance
   */
  ConnectivitySchemaGenerator build();
}
