/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.builder.adapter;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptySet;
import static java.util.Collections.sort;
import static java.util.Collections.unmodifiableList;
import static java.util.Comparator.comparing;
import static java.util.Optional.empty;
import static java.util.Optional.ofNullable;
import static java.util.function.Function.identity;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static org.mule.runtime.api.meta.ExpressionSupport.NOT_SUPPORTED;
import static org.mule.runtime.api.meta.model.parameter.ParameterRole.BEHAVIOUR;
import static org.mule.runtime.extension.api.util.NameUtils.getAliasName;

import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.ObjectFieldType;
import org.mule.metadata.api.model.ObjectType;
import org.mule.runtime.api.meta.NamedObject;
import org.mule.runtime.api.meta.model.ModelProperty;
import org.mule.runtime.api.meta.model.display.DisplayModel;
import org.mule.runtime.api.meta.model.display.LayoutModel;
import org.mule.runtime.api.meta.model.parameter.ExclusiveParametersModel;
import org.mule.runtime.api.meta.model.parameter.ParameterGroupModel;
import org.mule.runtime.api.meta.model.parameter.ParameterModel;
import org.mule.runtime.extension.api.declaration.type.annotation.ExclusiveOptionalsTypeAnnotation;
import org.mule.runtime.extension.api.declaration.type.annotation.FlattenedTypeAnnotation;
import org.mule.runtime.extension.api.declaration.type.annotation.QNameTypeAnnotation;
import org.mule.runtime.extension.api.model.parameter.ImmutableParameterModel;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Stream;

class ObjectTypeAsParameterGroupAdapter implements ParameterGroupModel {

  private final ObjectType adaptedType;
  private final String groupName;
  private final Map<String, ParameterModel> parameterModelsByName;
  private final List<ParameterModel> parameterModels;
  private final List<ExclusiveParametersModel> exclusiveParametersModel;

  public ObjectTypeAsParameterGroupAdapter(ObjectType adaptedType, MetadataType stringType) {
    this.adaptedType = adaptedType;
    this.groupName = resolveGroupName(adaptedType);

    List<ParameterModel> tempParameterModels = new ArrayList<>();
    List<ExclusiveParametersModel> tempExclusiveParametersModels = new ArrayList<>();

    for (ObjectFieldType wrappedFieldType : adaptedType.getFields()) {
      Stream<ObjectFieldType> fieldTypes;
      if (wrappedFieldType.getAnnotation(FlattenedTypeAnnotation.class).isPresent()
          && wrappedFieldType.getValue() instanceof ObjectType) {
        fieldTypes = ((ObjectType) (wrappedFieldType.getValue())).getFields().stream();
      } else {
        fieldTypes = Stream.of(wrappedFieldType);
      }

      tempParameterModels.addAll(fieldTypes
          .map(field -> new ObjectFieldTypeAsParameterModelAdapter(field, adaptedType.getAnnotation(QNameTypeAnnotation.class)
              .map(QNameTypeAnnotation::getValue)))
          .collect(toList()));

      wrappedFieldType.getAnnotation(ExclusiveOptionalsTypeAnnotation.class)
          .map(MetadataTypeExclusiveParametersModelAdapter::new)
          .ifPresent(tempExclusiveParametersModels::add);
    }

    sort(tempParameterModels, comparing(ParameterModel::getName));
    // Force a consistent order
    sort(tempExclusiveParametersModels, comparing(excl -> excl.getExclusiveParameterNames().iterator().next()));

    this.parameterModelsByName = tempParameterModels
        .stream()
        .collect(toMap(NamedObject::getName, identity()));

    if (!parameterModelsByName.containsKey("name")) {
      final ImmutableParameterModel nameParam = new ImmutableParameterModel("name", "The name of this object in the DSL",
                                                                            stringType,
                                                                            false, true, false, true, NOT_SUPPORTED,
                                                                            null, BEHAVIOUR, null, null, null, null,
                                                                            emptyList(), emptySet());
      this.parameterModelsByName.put("name", nameParam);
      tempParameterModels.add(nameParam);
    }

    this.parameterModels = unmodifiableList(tempParameterModels);
    this.exclusiveParametersModel = unmodifiableList(tempExclusiveParametersModels);
  }

  private String resolveGroupName(ObjectType adaptedType) {
    String aliasName;
    try {
      aliasName = getAliasName(adaptedType);
    } catch (IllegalArgumentException e) {
      aliasName = DEFAULT_GROUP_NAME;
    }
    return aliasName;
  }

  @Override
  public List<ParameterModel> getParameterModels() {
    return parameterModels;
  }

  @Override
  public Optional<ParameterModel> getParameter(String name) {
    return ofNullable(parameterModelsByName.get(name));
  }

  @Override
  public String getName() {
    return groupName;
  }

  @Override
  public String getDescription() {
    return "";
  }

  @Override
  public Optional<DisplayModel> getDisplayModel() {
    return empty();
  }

  @Override
  public Optional<LayoutModel> getLayoutModel() {
    return empty();
  }

  @Override
  public <T extends ModelProperty> Optional<T> getModelProperty(Class<T> propertyType) {
    return empty();
  }

  @Override
  public Set<ModelProperty> getModelProperties() {
    return emptySet();
  }

  @Override
  public List<ExclusiveParametersModel> getExclusiveParametersModels() {
    return exclusiveParametersModel;
  }

  @Override
  public boolean isShowInDsl() {
    return false;
  }

  @Override
  public String toString() {
    return "ObjectTypeAsParameterGroupAdapter{" + getName() + "; " + adaptedType.toString() + "}";
  }
}
