/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.ast.internal.builder;

import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.toList;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.ast.api.ImportedResource;
import org.mule.runtime.ast.api.NamespaceDefinition;
import org.mule.runtime.ast.api.builder.ArtifactAstBuilder;
import org.mule.runtime.ast.api.builder.ComponentAstBuilder;
import org.mule.runtime.ast.internal.DefaultArtifactAst;
import org.mule.runtime.ast.internal.model.ExtensionModelHelper;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.function.Supplier;
import java.util.function.UnaryOperator;

public class DefaultArtifactAstBuilder implements ArtifactAstBuilder {

  private final ExtensionModelHelper extModelHelper;
  private final Optional<ArtifactAst> parentArtifact;
  private final List<Supplier<ComponentAst>> topLevelComponents = new ArrayList<>();
  private final PropertiesResolver propertiesResolver = new PropertiesResolver();
  private final List<ImportedResource> importedResources = new ArrayList<>();

  private NamespaceDefinition namespaceDefinition;

  public DefaultArtifactAstBuilder(Set<ExtensionModel> extensionsModels, Optional<ArtifactAst> parentArtifact,
                                   UnaryOperator<String> basePropertiesResolver) {
    this.extModelHelper = new ExtensionModelHelper(extensionsModels);
    this.parentArtifact = parentArtifact;
    this.propertiesResolver.setMappingFunction(basePropertiesResolver);
  }

  @Override
  public ComponentAstBuilder addTopLevelComponent() {
    final DefaultComponentAstBuilder componentAstBuilder =
        new DefaultComponentAstBuilder(propertiesResolver, extModelHelper, emptyList(), topLevelComponents.size());
    topLevelComponents.add(componentAstBuilder);
    return componentAstBuilder;
  }

  public ArtifactAstBuilder addTopLevelComponent(ComponentAst topLevelComponent) {
    topLevelComponents.add(() -> topLevelComponent);
    return this;
  }

  @Override
  public void withNamespaceDefinition(NamespaceDefinition namespaceDefinition) {
    this.namespaceDefinition = namespaceDefinition;
  }

  @Override
  public ArtifactAstBuilder withImportedResource(ImportedResource resource) {
    resource.updatePropertiesResolver(propertiesResolver);
    this.importedResources.add(resource);
    return this;
  }

  @Override
  public ArtifactAst build() {
    final List<ComponentAst> builtTopLevel = topLevelComponents
        .stream()
        .map(Supplier::get)
        .collect(toList());

    return new DefaultArtifactAst(builtTopLevel, extModelHelper::getExtensionsModels, parentArtifact.orElse(null),
                                  propertiesResolver, namespaceDefinition, importedResources);
  }
}
