/*
 * Decompiled with CFR 0.152.
 */
package org.mule.util.store;

import java.io.File;
import java.io.FileFilter;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.OutputStream;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import org.apache.commons.collections.BidiMap;
import org.apache.commons.collections.bidimap.TreeBidiMap;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.mule.api.MuleContext;
import org.mule.api.MuleRuntimeException;
import org.mule.api.store.ExpirableObjectStore;
import org.mule.api.store.ListableObjectStore;
import org.mule.api.store.ObjectAlreadyExistsException;
import org.mule.api.store.ObjectDoesNotExistException;
import org.mule.api.store.ObjectStoreException;
import org.mule.api.store.ObjectStoreNotAvaliableException;
import org.mule.config.i18n.CoreMessages;
import org.mule.config.i18n.Message;
import org.mule.config.i18n.MessageFactory;
import org.mule.util.FileUtils;
import org.mule.util.SerializationUtils;
import org.mule.util.UUID;
import org.mule.util.store.DeserializationPostInitialisable;

public class PersistentObjectStorePartition<T extends Serializable>
implements ListableObjectStore<T>,
ExpirableObjectStore<T> {
    private static final String OBJECT_FILE_EXTENSION = ".obj";
    private static final String PARTITION_DESCRIPTOR_FILE = "partition-descriptor";
    protected final Log logger = LogFactory.getLog(this.getClass());
    private final MuleContext muleContext;
    private boolean loaded = false;
    private File partitionDirectory;
    private String partitionName;
    private final BidiMap realKeyToUUIDIndex = new TreeBidiMap();

    PersistentObjectStorePartition(MuleContext muleContext, String partitionName, File partitionDirectory) {
        this.muleContext = muleContext;
        this.partitionName = partitionName;
        this.partitionDirectory = partitionDirectory;
    }

    PersistentObjectStorePartition(MuleContext muleContext, File partitionDirectory) throws ObjectStoreNotAvaliableException {
        this.muleContext = muleContext;
        this.partitionDirectory = partitionDirectory;
        this.partitionName = this.readPartitionFileName(partitionDirectory);
    }

    private String readPartitionFileName(File partitionDirectory) throws ObjectStoreNotAvaliableException {
        File partitionDescriptorFile = new File(partitionDirectory, PARTITION_DESCRIPTOR_FILE);
        try {
            return FileUtils.readFileToString((File)partitionDescriptorFile);
        }
        catch (IOException e) {
            throw new ObjectStoreNotAvaliableException(e);
        }
    }

    @Override
    public synchronized void open() throws ObjectStoreException {
        this.createDirectory(this.partitionDirectory);
        this.createOrRetrievePartitionDescriptorFile();
    }

    @Override
    public void close() throws ObjectStoreException {
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public List<Serializable> allKeys() throws ObjectStoreException {
        this.assureLoaded();
        BidiMap bidiMap = this.realKeyToUUIDIndex;
        synchronized (bidiMap) {
            return Collections.unmodifiableList(new ArrayList(this.realKeyToUUIDIndex.keySet()));
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public boolean contains(Serializable key) throws ObjectStoreException {
        this.assureLoaded();
        BidiMap bidiMap = this.realKeyToUUIDIndex;
        synchronized (bidiMap) {
            return this.realKeyToUUIDIndex.containsKey((Object)key);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void store(Serializable key, T value) throws ObjectStoreException {
        this.assureLoaded();
        BidiMap bidiMap = this.realKeyToUUIDIndex;
        synchronized (bidiMap) {
            if (this.realKeyToUUIDIndex.containsKey((Object)key)) {
                throw new ObjectAlreadyExistsException();
            }
            File newFile = this.createFileToStoreObject();
            this.realKeyToUUIDIndex.put((Object)key, (Object)newFile.getName());
            this.serialize(newFile, new StoreValue<T>(key, value));
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void clear() throws ObjectStoreException {
        BidiMap bidiMap = this.realKeyToUUIDIndex;
        synchronized (bidiMap) {
            try {
                FileUtils.cleanDirectory(this.partitionDirectory);
            }
            catch (IOException e) {
                throw new ObjectStoreException(MessageFactory.createStaticMessage("Could not clear ObjectStore"), (Throwable)e);
            }
            this.realKeyToUUIDIndex.clear();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public T retrieve(Serializable key) throws ObjectStoreException {
        this.assureLoaded();
        BidiMap bidiMap = this.realKeyToUUIDIndex;
        synchronized (bidiMap) {
            if (!this.realKeyToUUIDIndex.containsKey((Object)key)) {
                String message = "Key does not exist: " + key;
                throw new ObjectDoesNotExistException(CoreMessages.createStaticMessage(message));
            }
            String filename = (String)this.realKeyToUUIDIndex.get((Object)key);
            File file = this.getValueFile(filename);
            return (T)((Serializable)this.deserialize(file).getValue());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public T remove(Serializable key) throws ObjectStoreException {
        this.assureLoaded();
        BidiMap bidiMap = this.realKeyToUUIDIndex;
        synchronized (bidiMap) {
            T value = this.retrieve(key);
            this.deleteStoreFile(this.getValueFile((String)this.realKeyToUUIDIndex.get((Object)key)));
            return value;
        }
    }

    @Override
    public boolean isPersistent() {
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void expire(int entryTTL, int maxEntries) throws ObjectStoreException {
        this.assureLoaded();
        BidiMap bidiMap = this.realKeyToUUIDIndex;
        synchronized (bidiMap) {
            Long lastModified;
            File[] files = this.listValuesFiles();
            Arrays.sort(files, new Comparator<File>(){

                @Override
                public int compare(File f1, File f2) {
                    int result = Long.valueOf(f1.lastModified()).compareTo(f2.lastModified());
                    if (result == 0) {
                        result = f1.getName().compareTo(f2.getName());
                    }
                    return result;
                }
            });
            int startIndex = this.trimToMaxSize(files, maxEntries);
            if (entryTTL == 0) {
                return;
            }
            long now = System.currentTimeMillis();
            for (int i = startIndex; i < files.length && now - (lastModified = Long.valueOf(files[i].lastModified())) >= (long)entryTTL; ++i) {
                this.deleteStoreFile(files[i]);
            }
        }
    }

    private void assureLoaded() throws ObjectStoreException {
        if (!this.loaded) {
            this.loadStoredKeysAndFileNames();
        }
    }

    private synchronized void loadStoredKeysAndFileNames() throws ObjectStoreException {
        if (this.loaded) {
            return;
        }
        try {
            File[] files = this.listValuesFiles();
            for (int i = 0; i < files.length; ++i) {
                File file = files[i];
                StoreValue<T> storeValue = this.deserialize(file);
                this.realKeyToUUIDIndex.put((Object)storeValue.getKey(), (Object)file.getName());
            }
            this.loaded = true;
        }
        catch (Exception e) {
            String message = String.format("Could not restore object store data from %1s", this.partitionDirectory.getAbsolutePath());
            throw new ObjectStoreException(CoreMessages.createStaticMessage(message));
        }
    }

    private File[] listValuesFiles() {
        File[] files = this.partitionDirectory.listFiles(new FileFilter(){

            @Override
            public boolean accept(File file) {
                return !file.isDirectory() && file.getName().endsWith(PersistentObjectStorePartition.OBJECT_FILE_EXTENSION);
            }
        });
        if (files == null) {
            files = new File[]{};
        }
        return files;
    }

    protected void createDirectory(File directory) throws ObjectStoreException {
        try {
            if (!directory.exists() && !directory.mkdirs()) {
                Message message = CoreMessages.failedToCreate("object store directory " + directory.getAbsolutePath());
                throw new MuleRuntimeException(message);
            }
        }
        catch (Exception e) {
            throw new ObjectStoreException(e);
        }
    }

    private File getValueFile(String filename) {
        return new File(this.partitionDirectory, filename);
    }

    protected File createFileToStoreObject() throws ObjectStoreException {
        String filename = UUID.getUUID() + OBJECT_FILE_EXTENSION;
        try {
            return FileUtils.newFile(this.partitionDirectory, filename);
        }
        catch (MuleRuntimeException mre) {
            throw new ObjectStoreException(mre);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected File createOrRetrievePartitionDescriptorFile() throws ObjectStoreException {
        try {
            File partitionDescriptorFile = new File(this.partitionDirectory, PARTITION_DESCRIPTOR_FILE);
            if (partitionDescriptorFile.exists()) {
                this.partitionName = this.readPartitionFileName(this.partitionDirectory);
                return partitionDescriptorFile;
            }
            try (FileWriter fileWriter = new FileWriter(partitionDescriptorFile.getAbsolutePath(), false);){
                fileWriter.write(this.partitionName);
                fileWriter.flush();
            }
            return partitionDescriptorFile;
        }
        catch (Exception e) {
            throw new ObjectStoreException(e);
        }
    }

    protected void serialize(File outputFile, StoreValue<T> storeValue) throws ObjectStoreException {
        FileOutputStream out = null;
        try {
            out = new FileOutputStream(outputFile);
            ObjectOutputStream objectOutputStream = new ObjectOutputStream(out);
            SerializationUtils.serialize(storeValue, (OutputStream)objectOutputStream);
        }
        catch (Exception se) {
            throw new ObjectStoreException(se);
        }
        finally {
            if (out != null) {
                try {
                    out.close();
                }
                catch (Exception e) {
                    this.logger.warn((Object)("error closing file " + outputFile.getAbsolutePath()));
                }
            }
        }
    }

    protected StoreValue<T> deserialize(File file) throws ObjectStoreException {
        ObjectInputStream objectInputStream = null;
        try {
            objectInputStream = new ObjectInputStream(new FileInputStream(file));
            StoreValue storedValue = (StoreValue)SerializationUtils.deserialize((InputStream)objectInputStream, this.muleContext);
            if (storedValue.getValue() instanceof DeserializationPostInitialisable) {
                DeserializationPostInitialisable.Implementation.init(storedValue.getValue(), this.muleContext);
            }
            StoreValue storeValue = storedValue;
            return storeValue;
        }
        catch (FileNotFoundException e) {
            throw new ObjectDoesNotExistException(e);
        }
        catch (Exception e) {
            throw new ObjectStoreException(e);
        }
        finally {
            if (objectInputStream != null) {
                try {
                    objectInputStream.close();
                }
                catch (Exception e) {
                    this.logger.warn((Object)("error closing opened file " + file.getAbsolutePath()));
                }
            }
        }
    }

    protected void deleteStoreFile(File file) throws ObjectStoreException {
        if (file.exists()) {
            if (!file.delete()) {
                Message message = CoreMessages.createStaticMessage("Deleting " + file.getAbsolutePath() + " failed");
                throw new ObjectStoreException(message);
            }
        } else {
            throw new ObjectDoesNotExistException();
        }
        this.realKeyToUUIDIndex.removeValue((Object)file.getName());
    }

    private int trimToMaxSize(File[] files, int maxEntries) throws ObjectStoreException {
        if (maxEntries == 0) {
            return 0;
        }
        int expired = 0;
        int excess = files.length - maxEntries;
        if (excess > 0) {
            for (int i = 0; i < excess; ++i) {
                this.deleteStoreFile(files[i]);
                ++expired;
            }
        }
        return expired;
    }

    public String getPartitionName() {
        return this.partitionName;
    }

    public static class StoreValue<T>
    implements Serializable {
        private Serializable key;
        private T value;

        public StoreValue(Serializable key, T value) {
            this.key = key;
            this.value = value;
        }

        public Serializable getKey() {
            return this.key;
        }

        public T getValue() {
            return this.value;
        }
    }
}

