/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.soap.internal.message;

import org.mule.soap.api.message.AddressingProperties;
import org.mule.soap.api.message.ReliableMessagingProperties;
import org.mule.soap.api.message.SoapAttachment;
import org.mule.soap.api.message.SoapRequest;

import java.io.InputStream;
import java.util.Map;
import java.util.Optional;

import static java.util.Collections.unmodifiableMap;
import static java.util.Optional.ofNullable;

/**
 * Default implementation of a {@link SoapRequest}.
 *
 * @since 1.0
 */
public final class DefaultSoapRequest implements SoapRequest {

  private final InputStream content;
  private final Map<String, String> soapHeaders;
  private final Map<String, SoapAttachment> attachments;
  private final Map<String, String> transportHeaders;
  private final String contentType;
  private final String operation;
  private final boolean useXMLInitialDeclaration;
  private final AddressingProperties addressingProperties;
  private final ReliableMessagingProperties reliableMessagingProperties;

  public DefaultSoapRequest(InputStream content,
                            Map<String, String> soapHeaders,
                            Map<String, String> transportHeaders,
                            Map<String, SoapAttachment> attachments,
                            String contentType,
                            String operation,
                            boolean useXMLInitialDeclaration,
                            AddressingProperties addressingProperties,
                            ReliableMessagingProperties reliableMessagingProperties) {
    this.content = content;
    this.soapHeaders = unmodifiableMap(soapHeaders);
    this.transportHeaders = unmodifiableMap(transportHeaders);
    this.attachments = unmodifiableMap(attachments);
    this.contentType = contentType;
    this.operation = operation;
    this.useXMLInitialDeclaration = useXMLInitialDeclaration;
    this.addressingProperties = addressingProperties;
    this.reliableMessagingProperties = reliableMessagingProperties;
  }

  @Override
  public InputStream getContent() {
    return content;
  }

  @Override
  public Map<String, String> getSoapHeaders() {
    return soapHeaders;
  }

  @Override
  public Map<String, String> getTransportHeaders() {
    return transportHeaders;
  }

  @Override
  public Map<String, SoapAttachment> getAttachments() {
    return attachments;
  }

  @Override
  public String getContentType() {
    return contentType;
  }

  @Override
  public String getOperation() {
    return operation;
  }

  @Override
  public boolean isUseXMLInitialDeclaration() {
    return useXMLInitialDeclaration;
  }

  @Override
  public Optional<AddressingProperties> getAddressingProperties() {
    return ofNullable(addressingProperties);
  }

  @Override
  public Optional<ReliableMessagingProperties> getReliableMessagingProperties() {
    return ofNullable(reliableMessagingProperties);
  }
}
