/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.soap.internal.interceptor;

import com.google.common.collect.ImmutableMap;
import org.apache.cxf.binding.soap.SoapHeader;
import org.apache.cxf.binding.soap.SoapMessage;
import org.apache.cxf.binding.soap.interceptor.AbstractSoapInterceptor;
import org.apache.cxf.interceptor.Fault;
import org.apache.cxf.message.Exchange;
import org.mule.soap.api.client.BadResponseException;
import org.mule.soap.api.message.SoapResponse;
import org.mule.soap.internal.client.AbstractSoapCxfClient;
import org.mule.soap.internal.util.XmlTransformationException;
import org.mule.soap.internal.util.XmlTransformationUtils;
import org.w3c.dom.Element;
import org.w3c.dom.Node;

import static java.lang.String.format;
import static org.apache.cxf.phase.Phase.PRE_PROTOCOL;

/**
 * CXF out interceptor that collects the received SOAP headers in the response, transforms it and stores them in the response
 * message {@link Exchange} so then can be returned in a {@link SoapResponse}.
 *
 * @since 1.0
 */
public class OutputSoapHeadersInterceptor extends AbstractSoapInterceptor {

  private final boolean ENABLE_NAMESPACES = Boolean.valueOf(System.getProperty(AbstractSoapCxfClient.MULE_ENABLE_NAMESPACES));

  public OutputSoapHeadersInterceptor() {
    super(PRE_PROTOCOL);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void handleMessage(SoapMessage message) throws Fault {
    ImmutableMap.Builder<String, String> headers = ImmutableMap.builder();
    // TODO: Using system property to include namespaces in headers. In future mayor version property should not be used and just
    // include namespace. Delete if and leave: headers.put(((ElementNSImpl) header.getObject()).getNodeName(),
    // getHeaderInputStream(header));
    message.getHeaders().stream()
        .filter(header -> header instanceof SoapHeader)
        .map(h -> (SoapHeader) h)
        .forEach(header -> {
          if (ENABLE_NAMESPACES) {
            headers.put(((Element) header.getObject()).getNodeName(),
                        getHeaderInputStream(header));
          } else {
            headers.put(header.getName().getLocalPart(), getHeaderInputStream(header));
          }
        });
    message.getExchange().put(AbstractSoapCxfClient.MULE_HEADERS_KEY, headers.build());
  }

  private String getHeaderInputStream(SoapHeader h) {
    try {
      return XmlTransformationUtils.nodeToString((Node) h.getObject());
    } catch (XmlTransformationException e) {
      throw new BadResponseException(format("Error while processing response header [%s]", h.getName()), e);
    }
  }
}
