/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.soap.internal.client;

import static org.apache.commons.lang3.StringUtils.isNotBlank;

import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import org.apache.cxf.endpoint.Client;
import org.jetbrains.annotations.NotNull;
import org.mule.soap.api.SoapWebServiceConfiguration;
import org.mule.soap.api.client.SoapClient;
import org.mule.soap.api.client.SoapClientFactory;
import org.mule.soap.api.client.SoapConfigurationException;
import org.mule.soap.api.transport.locator.TransportResourceLocator;
import org.mule.soap.internal.classloader.FipsClassLoaderDelegate;
import org.mule.wsdl.parser.WsdlParser;
import org.mule.wsdl.parser.locator.ResourceLocator;
import org.mule.wsdl.parser.model.PortModel;
import org.mule.wsdl.parser.model.ServiceModel;
import org.mule.wsdl.parser.model.WsdlModel;

import java.io.InputStream;

import static java.lang.String.format;
import static org.mule.wsdl.parser.model.WsdlStyle.RPC;

/**
 * {@link SoapClientFactory} implementation that creates {@link AbstractSoapCxfClient} instances.
 *
 * @since 1.0
 */
public class SoapCxfClientFactory implements SoapClientFactory {

  private CxfClientProvider cxfClientProvider = new CxfClientProvider();

  /**
   * Cache of {@link WsdlModel}. The keys are the wsdl locations.
   */
  private transient Cache<String, WsdlModel> wsdlModelCache = Caffeine.newBuilder().maximumSize(32).build();

  private static final FipsClassLoaderDelegate FIPS_CLASS_LOADER_DELEGATE =
      new FipsClassLoaderDelegate(SoapCxfClientFactory.class.getClassLoader());

  /**
   * Creates a new instance of a {@link AbstractSoapCxfClient} for the given address ans soap version.
   *
   * @throws SoapConfigurationException if there is a configuration error.
   */
  @Override
  public SoapClient create(SoapWebServiceConfiguration config) throws SoapConfigurationException {
    WsdlModel wsdl = getWsdlDefinition(config);
    Client client = cxfClientProvider.getClient(config, FIPS_CLASS_LOADER_DELEGATE);
    ServiceModel service = wsdl.getService(config.getService());
    if (service == null) {
      throw new SoapConfigurationException("Service [" + config.getService() + "] is not defined in the wsdl");
    }
    PortModel port = service.getPort(config.getPort());
    if (port == null) {
      throw new SoapConfigurationException("Port [" + config.getPort() + "] not found in service [" + service.getName() + "]");
    }
    String address = getAddress(config, port.getAddress());
    if (config.isMtomEnabled()) {
      return new SoapMtomCxfClient(client, wsdl, port, address, config.getEncoding());
    } else {
      return new DefaultSoapCxfClient(client, wsdl, port, address, config.getEncoding());
    }
  }

  private String getAddress(SoapWebServiceConfiguration config, String serviceAddress) throws SoapConfigurationException {

    String configAddress = config.getAddress();
    if (configAddress == null && serviceAddress == null) {
      throw new SoapConfigurationException("No address was specified and no one was found for the given configuration");
    }

    return isNotBlank(configAddress) ? configAddress : serviceAddress;
  }

  private WsdlModel getWsdlDefinition(SoapWebServiceConfiguration config) throws SoapConfigurationException {
    String location = config.getWsdlLocation();

    WsdlModel wsdlModel = wsdlModelCache
        .get(location,
             k -> WsdlParser.Companion.parse(location, new ResourceLocatorAdapter(config.getLocator()), config.getEncoding()));

    if (RPC.equals(wsdlModel.getStyle())) {
      throw new SoapConfigurationException(format("The provided WSDL [%s] is RPC style, RPC WSDLs are not supported", location));
    }
    return wsdlModel;
  }

  private static class ResourceLocatorAdapter implements ResourceLocator {

    private final TransportResourceLocator locator;

    ResourceLocatorAdapter(TransportResourceLocator locator) {
      this.locator = locator;
    }

    @Override
    public boolean handles(String s) {
      return locator.handles(s);
    }

    @NotNull
    @Override
    public InputStream getResource(String s) {
      return locator.getResource(s);
    }
  }

}
