// ========================================================================
// Copyright 2006 Mort Bay Consulting Pty. Ltd.
// ------------------------------------------------------------------------
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at 
// http://www.apache.org/licenses/LICENSE-2.0
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
// ========================================================================

package org.mortbay.cometd.continuation;

import java.util.TimerTask;
import java.util.concurrent.atomic.AtomicInteger;

import org.mortbay.cometd.AbstractBayeux;
import org.mortbay.cometd.ClientImpl;
import org.mortbay.util.ajax.Continuation;

/* ------------------------------------------------------------ */
/**
 * Extension of {@link ClientImpl} that uses {@link Continuation}s to
 * resume clients waiting for messages. Continuation clients are used for
 * remote clients and have removed if they are not accessed within
 * an idle timeout (@link {@link ContinuationBayeux#_clientTimer}).
 * 
 * @author gregw
 *
 */
class ContinuationClient extends ClientImpl
{
    private long _accessed;
    private transient TimerTask _timeout; 
    private ContinuationBayeux _bayeux;
    
    ContinuationClient(ContinuationBayeux bayeux)
    {
        super(bayeux,null,null);
        _bayeux=bayeux;
    }

    private transient Continuation _continuation;

    void setContinuation(Continuation continuation)
    {
        synchronized (this)
        {
            if (_continuation!=null && _continuation.isPending())
                _continuation.resume();
            _continuation=continuation;
        }
    }

    public void resume()
    {
        synchronized (this)
        {
            if (_continuation!=null)
                   _continuation.resume();
            _continuation=null;
        }
    }

    /* ------------------------------------------------------------ */
    public boolean isLocal()
    {
        return true;
    }

    /* ------------------------------------------------------------ */
    public void access()
    {
        synchronized(this)
        {
            // distribute access time in cluster
            _accessed=System.currentTimeMillis();
        
            if (_timeout!=null)
                _timeout.cancel();
            
            // a bit of a waste creating a new instance every time!
            _timeout = new TimerTask()
            {
                public void run()
                {
                    long now = System.currentTimeMillis();
                    if (_accessed+_bayeux.getClientTimeoutMs()<now)
                        remove(true);
                }
            };
                
            ((ContinuationBayeux)_bayeux)._clientTimer.schedule(_timeout,_bayeux.getClientTimeoutMs());
        }
        
    }
    
    /* (non-Javadoc)
     * @see org.mortbay.cometd.ClientImpl#remove(boolean)
     */
    public void remove(boolean timeout) 
    {
        if (timeout == false) 
        {
            _timeout.cancel();
            _timeout = null;
        }
        super.remove(timeout);
    }

}