/* Copyright (c) 2023 LibJ
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * You should have received a copy of The MIT License (MIT) along with this
 * program. If not, see <http://opensource.org/licenses/MIT/>.
 */

package org.libj.util;

import java.util.Comparator;
import java.util.Map;
import java.util.SortedMap;

/**
 * A {@link DelegateMap} that provides callback methods to observe the retrieval, addition, and removal of elements, either due to
 * direct method invocation on the map instance itself, or via {@link #entrySet()}, {@link #values()},
 * {@link #forEach(java.util.function.BiConsumer)}, and any other entrypoint that facilitates modification of the elements in this
 * map.
 *
 * @param <K> The type of keys maintained by this map.
 * @param <V> The type of mapped values.
 * @see #beforeGet(Object)
 * @see #afterGet(Object,Object,RuntimeException)
 * @see #beforePut(Object,Object,Object,Object)
 * @see #afterPut(Object,Object,Object,RuntimeException)
 * @see #beforeRemove(Object,Object)
 * @see #afterRemove(Object,Object,RuntimeException)
 */
public abstract class ObservableSortedMap<K,V> extends ObservableMap<K,V> implements SortedMap<K,V> {
  /**
   * Creates a new {@link ObservableSortedMap} with the specified target {@link Map}.
   *
   * @param map The target {@link SortedMap}.
   * @throws NullPointerException If {@code map} is null.
   */
  public ObservableSortedMap(final SortedMap<K,V> map) {
    super(map);
  }

  @Override
  public Comparator<? super K> comparator() {
    return ((SortedMap<K,V>)target).comparator();
  }

  protected class ObservableSortedSubMap extends ObservableSortedMap<K,V> {
    protected ObservableSortedSubMap(final SortedMap<K,V> map) {
      super(map);
    }

    @Override
    protected boolean beforeContainsKey(final Object key) {
      return ObservableSortedMap.this.beforeContainsKey(key);
    }

    @Override
    protected void afterContainsKey(final Object key, final boolean result, final RuntimeException e) {
      ObservableSortedMap.this.afterContainsKey(key, result, e);
    }

    @Override
    protected void beforeContainsValue(final Object value) {
      ObservableSortedMap.this.beforeContainsValue(value);
    }

    @Override
    protected void afterContainsValue(final Object value, final boolean result, final RuntimeException e) {
      ObservableSortedMap.this.afterContainsValue(value, result, e);
    }

    @Override
    protected Object beforeGet(final Object key) {
      return ObservableSortedMap.this.beforeGet(key);
    }

    @Override
    protected V afterGet(final Object key, final V value, final RuntimeException e) {
      return ObservableSortedMap.this.afterGet(key, value, e);
    }

    @Override
    protected Object beforePut(final K key, final V oldValue, final V newValue, final Object preventDefault) {
      return ObservableSortedMap.this.beforePut(key, oldValue, newValue, preventDefault);
    }

    @Override
    protected void afterPut(final K key, final V oldValue, final V newValue, final RuntimeException e) {
      ObservableSortedMap.this.afterPut(key, oldValue, newValue, e);
    }

    @Override
    protected boolean beforeRemove(final Object key, final V value) {
      return ObservableSortedMap.this.beforeRemove(key, value);
    }

    @Override
    protected void afterRemove(final Object key, final V value, final RuntimeException e) {
      ObservableSortedMap.this.afterRemove(key, value, e);
    }
  }

  @Override
  public SortedMap<K,V> subMap(final K fromKey, final K toKey) {
    return new ObservableSortedSubMap(((SortedMap<K,V>)target).subMap(fromKey, toKey));
  }

  @Override
  public SortedMap<K,V> headMap(final K toKey) {
    return new ObservableSortedSubMap(((SortedMap<K,V>)target).headMap(toKey));
  }

  @Override
  public SortedMap<K,V> tailMap(final K fromKey) {
    return new ObservableSortedSubMap(((SortedMap<K,V>)target).tailMap(fromKey));
  }

  @Override
  public K firstKey() {
    return ((SortedMap<K,V>)target).firstKey();
  }

  @Override
  public K lastKey() {
    return ((SortedMap<K,V>)target).lastKey();
  }
}