/* Copyright (c) 2023 LibJ
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * You should have received a copy of The MIT License (MIT) along with this
 * program. If not, see <http://opensource.org/licenses/MIT/>.
 */

package org.libj.util;

import java.util.AbstractMap;
import java.util.Comparator;
import java.util.Map;
import java.util.SortedMap;

/**
 * A {@link DelegateSortedMap} contains some other {@link Map}, to which it delegates its method calls, possibly transforming the
 * data along the way or providing additional functionality. The class {@link DelegateSortedMap} itself simply overrides all methods
 * of {@link AbstractMap} with versions that pass all requests to the target {@link Map}. Subclasses of {@link DelegateSortedMap}
 * may further override some of these methods and may also provide additional methods and fields.
 *
 * @param <K> The type of keys maintained by this map.
 * @param <V> The type of mapped values.
 */
public abstract class DelegateSortedMap<K,V> extends DelegateMap<K,V> implements SortedMap<K,V> {
  /**
   * Creates a new {@link DelegateSortedMap} with the specified target {@link Map}.
   *
   * @param target The target {@link SortedMap}.
   * @throws NullPointerException If the target {@link SortedMap} is null.
   */
  public DelegateSortedMap(final SortedMap<K,V> target) {
    super(target);
  }

  /**
   * Creates a new {@link DelegateSortedMap} with a null target.
   */
  protected DelegateSortedMap() {
  }

  @Override
  public Comparator<? super K> comparator() {
    return ((SortedMap<K,V>)target).comparator();
  }

  @Override
  public SortedMap<K,V> subMap(final K fromKey, final K toKey) {
    return ((SortedMap<K,V>)target).subMap(fromKey, toKey);
  }

  @Override
  public SortedMap<K,V> headMap(final K toKey) {
    return ((SortedMap<K,V>)target).headMap(toKey);
  }

  @Override
  public SortedMap<K,V> tailMap(final K fromKey) {
    return ((SortedMap<K,V>)target).tailMap(fromKey);
  }

  @Override
  public K firstKey() {
    return ((SortedMap<K,V>)target).firstKey();
  }

  @Override
  public K lastKey() {
    return ((SortedMap<K,V>)target).lastKey();
  }
}