/* Copyright (c) 2020 LibJ
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * You should have received a copy of The MIT License (MIT) along with this
 * program. If not, see <http://opensource.org/licenses/MIT/>.
 */

package org.libj.util;

import java.util.NoSuchElementException;
import java.util.function.Supplier;

import javax.annotation.Generated;

import org.libj.util.function.CharConsumer;
import org.libj.util.function.CharSupplier;

/**
 * A container object which may or may not contain a {@code char} value. If a value is present, {@code isPresent()} returns
 * {@code true}. If no value is present, the object is considered <i>empty</i> and {@code isPresent()} returns {@code false}.
 * <p>
 * Additional methods that depend on the presence or absence of a contained value are provided, such as {@link #orElse(char)
 * orElse()} (returns a default value if no value is present) and {@link #ifPresent(CharConsumer) ifPresent()} (performs an action if
 * a value is present).
 * <p>
 * This is a value-based class; use of identity-sensitive operations (including reference equality ({@code ==}), identity hash code,
 * or synchronization) on instances of {@code OptionalChar} may have unpredictable results and should be avoided.
 *
 * @implNote {@code OptionalChar} is primarily intended for use as a method return type where there is a clear need to represent "no
 *           result." A variable whose type is {@code OptionalChar} should never itself be {@code null}; it should always point to an
 *           {@code OptionalChar} instance.
 */
@Generated(value="org.openjax.codegen.template.Templates", date="2024-02-27T13:50:20.763")
public final class OptionalChar {
  /** Common instance for {@code empty()}. */
  private static final OptionalChar EMPTY = new OptionalChar();

  /** If true then the value is present, otherwise indicates no value is present. */
  private final boolean isPresent;
  private final char value;

  /**
   * Construct an empty instance.
   *
   * @implNote Generally only one empty instance, {@link OptionalChar#EMPTY}, should exist per VM.
   */
  private OptionalChar() {
    this.isPresent = false;
    this.value = 0;
  }

  /**
   * Returns an empty {@link OptionalChar} instance. No value is present for this {@code OptionalChar}.
   *
   * @implNote Though it may be tempting to do so, avoid testing if an object is empty by comparing with {@code ==} against
   *           instances returned by {@code OptionalChar.empty()}. There is no guarantee that it is a singleton. Instead, use
   *           {@link #isPresent()}.
   * @return An empty {@code OptionalChar}.
   */
  public static OptionalChar empty() {
    return EMPTY;
  }

  /**
   * Construct an instance with the described value.
   *
   * @param value The char value to describe.
   */
  private OptionalChar(final char value) {
    this.isPresent = true;
    this.value = value;
  }

  /**
   * Returns an {@link OptionalChar} describing the given value.
   *
   * @param value The value to describe.
   * @return An {@link OptionalChar} with the value present.
   */
  public static OptionalChar of(final char value) {
    return new OptionalChar(value);
  }

  /**
   * If a value is present, returns the value, otherwise throws {@link NoSuchElementException}.
   *
   * @implNote The preferred alternative to this method is {@link #orElseThrow()}.
   * @return The value described by this {@link OptionalChar}.
   * @throws NoSuchElementException If no value is present.
   */
  public char getAsChar() {
    if (isPresent)
      return value;

    throw new NoSuchElementException("No value present");
  }

  /**
   * If a value is present, returns {@code true}, otherwise {@code false}.
   *
   * @return {@code true} if a value is present, otherwise {@code false}.
   */
  public boolean isPresent() {
    return isPresent;
  }

  /**
   * If a value is not present, returns {@code true}, otherwise {@code false}.
   *
   * @return {@code true} if a value is not present, otherwise {@code false}.
   */
  public boolean isEmpty() {
    return !isPresent;
  }

  /**
   * If a value is present, performs the given action with the value, otherwise does nothing.
   *
   * @param action The action to be performed, if a value is present.
   * @throws NullPointerException If value is present and the given action is null.
   */
  public void ifPresent(final CharConsumer action) {
    if (isPresent)
      action.accept(value);
  }

  /**
   * If a value is present, performs the given action with the value, otherwise performs the given empty-based action.
   *
   * @param action The action to be performed, if a value is present.
   * @param emptyAction The empty-based action to be performed, if no value is present.
   * @throws NullPointerException If a value is present and the given action is null, or no value is present and the given
   *           empty-based action is null.
   */
  public void ifPresentOrElse(final CharConsumer action, final Runnable emptyAction) {
    if (isPresent)
      action.accept(value);
    else
      emptyAction.run();
  }

//  /**
//   * If a value is present, returns a sequential {@link CharStream} containing only that value, otherwise returns an empty
//   * {@code CharStream}.
//   * <p>
//   * <b>Note:</b> This method can be used to transform a {@code Stream} of optional chars to an {@code CharStream} of present
//   * chars:
//   *
//   * <pre>
//   * {@code
//   *     Stream<OptionalChar> os = ..
//   *     CharStream s = os.flatMapToChar(OptionalChar::stream)
//   * }
//   * </pre>
//   *
//   * @return The optional value as an {@code CharStream}.
//   */
//  public CharStream stream() {
//    return isPresent ? CharStream.of(value) : CharStream.empty();
//  }

  /**
   * If a value is present, returns the value, otherwise returns {@code other}.
   *
   * @param other The value to be returned, if no value is present.
   * @return The value, if present, otherwise {@code other}.
   */
  public char orElse(final char other) {
    return isPresent ? value : other;
  }

  /**
   * If a value is present, returns the value, otherwise returns the result produced by the supplying function.
   *
   * @param supplier The supplying function that produces a value to be returned.
   * @return The value, if present, otherwise the result produced by the supplying function.
   * @throws NullPointerException If no value is present and the supplying function is null.
   */
  public char orElseGet(final CharSupplier supplier) {
    return isPresent ? value : supplier.getAsChar();
  }

  /**
   * If a value is present, returns the value, otherwise throws {@link NoSuchElementException}.
   *
   * @return The value described by this {@code OptionalChar}.
   * @throws NoSuchElementException If no value is present.
   */
  public char orElseThrow() {
    if (isPresent)
      return value;

    throw new NoSuchElementException("No value present");
  }

  /**
   * If a value is present, returns the value, otherwise throws an exception produced by the exception supplying function.
   *
   * @implNote A method reference to the exception constructor with an empty argument list can be used as the supplier. For example,
   *           {@link IllegalStateException#IllegalStateException() IllegalStateException::new}.
   * @param <T> Type of the exception to be thrown
   * @param exceptionSupplier The supplying function that produces an exception to be thrown.
   * @return The value, if present
   * @throws T If no value is present.
   * @throws NullPointerException If no value is present and the exception supplying function is null.
   */
  public <T extends Throwable> char orElseThrow(final Supplier<? extends T> exceptionSupplier) throws T {
    if (isPresent)
      return value;

    throw exceptionSupplier.get();
  }

  /**
   * Indicates whether some other object is "equal to" this {@link OptionalChar}. The other object is considered equal if:
   * <ul>
   * <li>it is also an {@code OptionalChar} and;
   * <li>both instances have no value present or;
   * <li>the present values are "equal to" each other via {@code ==}.
   * </ul>
   *
   * @param obj An object to be tested for equality.
   * @return {@code true} if the other object is "equal to" this object otherwise {@code false}.
   */
  @Override
  public boolean equals(final Object obj) {
    if (this == obj)
      return true;

    if (!(obj instanceof OptionalChar))
      return false;

    final OptionalChar that = (OptionalChar)obj;
    return isPresent && that.isPresent ? value == that.value : isPresent == that.isPresent;
  }

  /**
   * Returns the hash code of the value, if present, otherwise {@code 0} (zero) if no value is present.
   *
   * @return Hash code value of the present value or {@code 0} if no value is present.
   */
  @Override
  public int hashCode() {
    return isPresent ? Character.hashCode(value) : 0;
  }

  /**
   * Returns a non-empty string representation of this {@code OptionalChar} suitable for debugging. The exact presentation format is
   * unspecified and may vary between implementations and versions.
   *
   * @implNote If a value is present the result must include its string representation in the result. Empty and present
   *           {@code OptionalChar}s must be unambiguously differentiable.
   * @return The string representation of this instance.
   */
  @Override
  public String toString() {
    return isPresent ? String.format("OptionalChar[%s]", value) : "OptionalChar.empty";
  }
}