/* Copyright (c) 2018 LibJ
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * You should have received a copy of The MIT License (MIT) along with this
 * program. If not, see <http://opensource.org/licenses/MIT/>.
 */

package org.libj.util.primitive;

import static org.junit.Assert.*;

import java.util.Arrays;
import java.util.HashSet;
import java.util.NoSuchElementException;
import java.util.Random;
import java.util.Set;

import javax.annotation.Generated;

import org.junit.Assert;
import org.junit.Test;
import org.libj.lang.Strings;
import org.libj.util.CollectionUtil;

@SuppressWarnings("all")
@Generated(value="org.openjax.codegen.template.Templates", date="2024-02-27T13:50:20.763")
public class HashCharSetTest extends PrimitiveCollectionTest {
  private static final int INITIAL_CAPACITY = 64;

  private final HashCharSet testSet = new HashCharSet(INITIAL_CAPACITY);

  @Test
  public void initiallyContainsNoElements() {
    for (char i = 0; i < 100; ++i) // [N]
      assertFalse(testSet.contains(i));
  }

  @Test
  public void initiallyContainsNoBoxedElements() {
    for (char i = 0; i < 100; ++i) // [N]
      assertFalse(testSet.contains(Character.valueOf(i)));
  }

  @Test
  public void containsAddedElement() {
    assertTrue(testSet.add((char)1));
    assertTrue(testSet.contains((char)1));
  }

  @Test
  public void addingAnElementTwiceDoesNothing() {
    assertTrue(testSet.add((char)1));
    assertFalse(testSet.add((char)1));
  }

  @Test
  public void containsAddedBoxedElements() {
    assertTrue(testSet.add((char)1));
    assertTrue(testSet.add(Character.valueOf((char)2)));

    assertTrue(testSet.contains(Character.valueOf((char)1)));
    assertTrue(testSet.contains((char)2));
  }

  @Test
  public void removingAnElementFromAnEmptyListDoesNothing() {
    assertFalse(testSet.remove((char)0));
  }

  @Test
  public void removingAPresentElementRemovesIt() {
    assertTrue(testSet.add((char)1));
    assertTrue(testSet.remove((char)1));
    assertFalse(testSet.contains((char)1));
  }

  @Test
  public void sizeIsInitiallyZero() {
    assertEquals(0, testSet.size());
  }

  @Test
  public void sizeIncrementsWithNumberOfAddedElements() {
    addTwoElements(testSet);

    assertEquals(2, testSet.size());
  }

  @Test
  public void sizeContainsNumberOfNewElements() {
    testSet.add((char)1);
    testSet.add((char)1);

    assertEquals(1, testSet.size());
  }

  @Test
  public void iteratorsListElements() {
    addTwoElements(testSet);

    assertIteratorHasElements();
  }

  @Test
  public void iteratorsStartFromTheBeginningEveryTime() {
    iteratorsListElements();

    assertIteratorHasElements();
  }

  @Test
  public void iteratorsListElementsWithoutHasNext() {
    addTwoElements(testSet);

    assertIteratorHasElementsWithoutHasNext();
  }

  @Test
  public void iteratorsStartFromTheBeginningEveryTimeWithoutHasNext() {
    iteratorsListElementsWithoutHasNext();

    assertIteratorHasElementsWithoutHasNext();
  }

  @Test(expected = NoSuchElementException.class)
  public void iteratorsThrowNoSuchElementException() {
    addTwoElements(testSet);

    exhaustIterator();
  }

  @Test(expected = NoSuchElementException.class)
  public void iteratorsThrowNoSuchElementExceptionFromTheBeginningEveryTime() {
    addTwoElements(testSet);

    try {
      exhaustIterator();
    }
    catch (final NoSuchElementException ignore) {
    }

    exhaustIterator();
  }

  @Test
  public void iteratorHasNoElements() {
    assertFalse(testSet.iterator().hasNext());
  }

  @Test(expected = NoSuchElementException.class)
  public void iteratorThrowExceptionForEmptySet() {
    testSet.iterator().next();
  }

  @Test
  public void clearRemovesAllElementsOfTheSet() {
    addTwoElements(testSet);

    testSet.clear();

    assertEquals(0, testSet.size());
    assertFalse(testSet.contains((char)1));
    assertFalse(testSet.contains((char)101));
  }

  @Test
  public void differenceReturnsNullIfBothSetsEqual() {
    addTwoElements(testSet);

    final HashCharSet other = new HashCharSet(100);
    addTwoElements(other);

    other.removeAll(testSet);
    assertEquals(0, other.size());
  }

  @Test
  public void differenceReturnsSetDifference() {
    addTwoElements(testSet);

    final HashCharSet other = new HashCharSet(100);
    other.add((char)1);

    testSet.removeAll(other);
    assertTrue(testSet.contains((char)101));
  }

  @Test
  public void copiesOtherHashCharSet() {
    addTwoElements(testSet);

    final HashCharSet other = new HashCharSet(testSet);
    assertContainsElements(other);
  }

  @Test
  public void twoEmptySetsAreEqual() {
    Assert.assertEquals(testSet, new HashCharSet(100));
  }

  @Test
  public void setsWithTheSameValuesAreEqual() {
    final HashCharSet that = new HashCharSet(100);

    addTwoElements(testSet);
    addTwoElements(that);

    Assert.assertEquals(testSet, that);
  }

  @Test
  public void setsWithTheDifferentSizesAreNotEqual() {
    final HashCharSet that = new HashCharSet(100);

    addTwoElements(testSet);
    that.add((char)101);

    assertNotEquals(testSet, that);
  }

  @Test
  public void setsWithTheDifferentValuesAreNotEqual() {
    final HashCharSet that = new HashCharSet(100);

    addTwoElements(testSet);
    that.add((char)2);
    that.add((char)101);

    assertNotEquals(testSet, that);
  }

  @Test
  public void twoEmptySetsHaveTheSameHashcode() {
    assertEquals(testSet.hashCode(), new HashCharSet(100).hashCode());
  }

  @Test
  public void setsWithTheSameValuesHaveTheSameHashcode() {
    final HashCharSet other = new HashCharSet(100);

    addTwoElements(testSet);
    addTwoElements(other);

    assertEquals(testSet.hashCode(), other.hashCode());
  }

  @Test
  public void reducesSizeWhenElementRemoved() {
    addTwoElements(testSet);
    testSet.remove((char)101);

    assertEquals(1, testSet.size());
  }

  @Test(expected = NullPointerException.class)
  public void toArrayThrowsNullPointerExceptionForNullArgument() {
    final Character[] a = null;
    testSet.toArray(a);
  }

  @Test
  public void toArrayCopiesElementsCharacterToSufficientlySizedArray() {
    addTwoElements(testSet);
    final Character[] result = testSet.toArray(new Character[testSet.size()]);

    assertArrayContainingElements(result);
  }

  @Test
  public void toArrayCopiesElementsCharacterToNewArray() {
    addTwoElements(testSet);
    final Character[] result = testSet.toArray(new Character[testSet.size()]);

    assertArrayContainingElements(result);
  }

  @Test
  public void toArraySupportsEmptyCollection() {
    final Character[] result = testSet.toArray(new Character[testSet.size()]);

    Assert.assertArrayEquals(result, new Character[] {});
  }

  // Test case from usage bug.
  @Test
  public void chainCompactionShouldNotCauseElementsToBeMovedBeforeTheirHash() {
    final HashCharSet requiredFields = new HashCharSet(14);

    requiredFields.add((char)8);
    requiredFields.add((char)9);
    requiredFields.add((char)35);
    requiredFields.add((char)49);
    requiredFields.add((char)56);

    assertTrue("Failed to remove 8", requiredFields.remove((char)8));
    assertTrue("Failed to remove 9", requiredFields.remove((char)9));

    assertTrue(requiredFields.containsAll(Arrays.asList((char)35, (char)49, (char)56)));
  }

  @Test
  public void shouldResizeWhenItHitsCapacity() {
    for (char i = 0; i < 2 * INITIAL_CAPACITY - 1; ++i) // [N]
      assertTrue(testSet.add(i));

    for (char i = 0; i < 2 * INITIAL_CAPACITY - 1; ++i) // [N]
      assertTrue(testSet.contains(i));
  }

  @Test
  public void containsEmptySet() {
    assertTrue(testSet.containsAll(new HashCharSet(100)));
  }

  @Test
  public void containsSubset() {
    addTwoElements(testSet);

    final HashCharSet subset = new HashCharSet(100);
    subset.add((char)1);

    assertTrue(testSet.containsAll(subset));
  }

  @Test
  public void doesNotContainDisjointSet() {
    addTwoElements(testSet);

    final HashCharSet disjoint = new HashCharSet(100);
    disjoint.add((char)1);
    disjoint.add((char)102);

    assertFalse(testSet.containsAll(disjoint));
  }

  @Test
  public void doesNotContainSuperset() {
    addTwoElements(testSet);

    final HashCharSet superset = new HashCharSet(100);
    addTwoElements(superset);
    superset.add((char)15);

    assertFalse(testSet.containsAll(superset));
  }

  @Test
  public void addingEmptySetDoesNothing() {
    addTwoElements(testSet);

    assertFalse(testSet.addAll(new HashCharSet(100)));
    assertFalse(testSet.addAll(new HashSet<>()));
    assertContainsElements(testSet);
  }

  @Test
  public void containsValuesAddedFromDisjointSetPrimitive() {
    addTwoElements(testSet);

    final HashCharSet disjoint = new HashCharSet(100);

    disjoint.add((char)2);
    disjoint.add((char)102);

    assertTrue(testSet.addAll(disjoint));
    assertTrue(testSet.contains((char)1));
    assertTrue(testSet.contains((char)101));
    assertTrue(testSet.containsAll(disjoint));
  }

  @Test
  public void containsValuesAddedFromDisjointSet() {
    addTwoElements(testSet);

    final HashSet<Character> disjoint = new HashSet<>();

    disjoint.add((char)2);
    disjoint.add((char)102);

    assertTrue(testSet.addAll(disjoint));
    assertTrue(testSet.contains((char)1));
    assertTrue(testSet.contains((char)101));
    assertTrue(testSet.containsAll(disjoint));
  }

  @Test
  public void containsValuesAddedFromCharacterersectingSetPrimitive() {
    addTwoElements(testSet);

    final HashCharSet intersecting = new HashCharSet(100);
    intersecting.add((char)1);
    intersecting.add((char)102);

    assertTrue(testSet.addAll(intersecting));
    assertTrue(testSet.contains((char)1));
    assertTrue(testSet.contains((char)101));
    assertTrue(testSet.containsAll(intersecting));
  }

  @Test
  public void containsValuesAddedFromCharacterersectingSet() {
    addTwoElements(testSet);

    final HashSet<Character> intersecting = new HashSet<>();

    intersecting.add((char)1);
    intersecting.add((char)102);

    assertTrue(testSet.addAll(intersecting));
    assertTrue(testSet.contains((char)1));
    assertTrue(testSet.contains((char)101));
    assertTrue(testSet.containsAll(intersecting));
  }

  @Test
  public void removingEmptySetDoesNothing() {
    addTwoElements(testSet);

    assertFalse(testSet.removeAll(new HashCharSet(100)));
    assertFalse(testSet.removeAll(new HashSet<>()));
    assertContainsElements(testSet);
  }

  @Test
  public void removingDisjointSetDoesNothing() {
    addTwoElements(testSet);

    final HashCharSet disjoint = new HashCharSet(100);
    disjoint.add((char)2);
    disjoint.add((char)102);

    assertFalse(testSet.removeAll(disjoint));
    assertFalse(testSet.removeAll(new HashSet<>()));
    assertContainsElements(testSet);
  }

  @Test
  public void doesNotContainRemovedCharacterersectingSetPrimitive() {
    addTwoElements(testSet);

    final HashCharSet intersecting = new HashCharSet(100);

    intersecting.add((char)1);
    intersecting.add((char)102);

    assertTrue(testSet.removeAll(intersecting));
    assertTrue(testSet.contains((char)101));
    assertFalse(testSet.containsAll(intersecting));
  }

  @Test
  public void doesNotContainRemovedCharacterersectingSet() {
    addTwoElements(testSet);

    final HashSet<Character> intersecting = new HashSet<>();
    intersecting.add((char)1);
    intersecting.add((char)102);

    assertTrue(testSet.removeAll(intersecting));
    assertTrue(testSet.contains((char)101));
    assertFalse(testSet.containsAll(intersecting));
  }

  @Test
  public void isEmptyAfterRemovingEqualSetPrimitive() {
    addTwoElements(testSet);

    final HashCharSet equal = new HashCharSet(100);
    addTwoElements(equal);

    assertTrue(testSet.removeAll(equal));
    assertTrue(testSet.isEmpty());
  }

  @Test
  public void isEmptyAfterRemovingEqualSet() {
    addTwoElements(testSet);

    final HashSet<Character> equal = new HashSet<>();
    addTwoElements(equal);

    assertTrue(testSet.removeAll(equal));
    assertTrue(testSet.isEmpty());
  }

  @Test
  public void removeElementsFromIterator() {
    addTwoElements(testSet);

    final CharIterator iterator = testSet.iterator();
    while (iterator.hasNext())
      if (iterator.next() == 1)
        iterator.remove();

    assertEquals(1, testSet.size());
    assertTrue(testSet.contains((char)101));
  }

  @Test
  public void shouldNotContainMissingValueInitially() {
    assertFalse(testSet.contains(HashCharSet.NULL));
  }

  @Test
  public void shouldAllowMissingValue() {
    assertTrue(testSet.add(HashCharSet.NULL));
    assertTrue(testSet.contains(HashCharSet.NULL));
    assertFalse(testSet.add(HashCharSet.NULL));
  }

  @Test
  public void shouldAllowRemovalOfMissingValue() {
    assertTrue(testSet.add(HashCharSet.NULL));
    assertTrue(testSet.remove(HashCharSet.NULL));
    assertFalse(testSet.contains(HashCharSet.NULL));
    assertFalse(testSet.remove(HashCharSet.NULL));
  }

  @Test
  public void sizeAccountsForMissingValue() {
    testSet.add((char)1);
    testSet.add(HashCharSet.NULL);

    assertEquals(2, testSet.size());
  }

  @Test
  public void toArrayCopiesElementsCharacterToNewArrayIncludingMissingValue() {
    addTwoElements(testSet);
    testSet.add(HashCharSet.NULL);

    final Character[] result = testSet.toArray(new Character[testSet.size()]);
    assertTrue(Arrays.asList(result).containsAll(Arrays.asList((char)1, (char)101, HashCharSet.NULL)));
  }

  @Test
  public void toObjectArrayCopiesElementsCharacterToNewArrayIncludingMissingValue() {
    addTwoElements(testSet);
    testSet.add(HashCharSet.NULL);

    final char[] result = testSet.toArray();
    Arrays.sort(result);
    assertArrayEquals(new char[] {HashCharSet.NULL, 1, 101}, result);
  }

  @Test
  public void equalsAccountsForMissingValue() {
    addTwoElements(testSet);
    testSet.add(HashCharSet.NULL);

    final HashCharSet other = new HashCharSet(100);
    addTwoElements(other);

    assertNotEquals(testSet, other);

    other.add(HashCharSet.NULL);
    Assert.assertEquals(testSet, other);

    testSet.remove(HashCharSet.NULL);

    assertNotEquals(testSet, other);
  }

  @Test
  public void consecutiveValuesShouldBeCorrectlyStored() {
    for (char i = 0; i < 127; ++i) // [N]
      testSet.add(i);

    assertEquals(127, testSet.size());

    int distinctElements = 0;
    for (final CharIterator i = testSet.iterator(); i.hasNext(); i.next()) // [I]
      ++distinctElements;

    assertEquals(distinctElements, 127);
  }

  @Test
  public void hashCodeAccountsForMissingValue() {
    addTwoElements(testSet);
    testSet.add(HashCharSet.NULL);

    final HashCharSet other = new HashCharSet(100);
    addTwoElements(other);

    other.add(HashCharSet.NULL);
    assertEquals(testSet.hashCode(), other.hashCode());
  }

  @Test
  public void iteratorAccountsForMissingValue() {
    addTwoElements(testSet);
    testSet.add(HashCharSet.NULL);

    int missingValueCount = 0;
    final CharIterator iterator = testSet.iterator();
    while (iterator.hasNext())
      if (iterator.next() == HashCharSet.NULL)
        ++missingValueCount;

    assertEquals(1, missingValueCount);
  }

  @Test
  public void iteratorCanRemoveMissingValue() {
    addTwoElements(testSet);
    testSet.add(HashCharSet.NULL);

    final CharIterator iterator = testSet.iterator();
    while (iterator.hasNext())
      if (iterator.next() == HashCharSet.NULL)
        iterator.remove();

    assertFalse(testSet.contains(HashCharSet.NULL));
  }

  @Test
  public void shouldGenerateStringRepresentation() {
    final char[] testEntries = {(char)65, (char)68, (char)83, (char)104, (char)111, (char)75, (char)83, (char)97};
    for (final char testEntry : testEntries) // [A]
      testSet.add(testEntry);

    final String string = testSet.toString();
    final String[] parts = Strings.split(string.substring(1, string.length() - 1).replace(" ", ""), ',');
    final HashSet<String> strings = CollectionUtil.asCollection(new HashSet<>(testSet.size()), parts);
    for (final char testEntry : testEntries) // [A]
      assertTrue(Arrays.toString(parts), strings.contains(String.valueOf(testEntry)));
  }

  @Test
  public void shouldRemoveMissingValueWhenCleared() {
    assertTrue(testSet.add(HashCharSet.NULL));

    testSet.clear();

    assertFalse(testSet.contains(HashCharSet.NULL));
  }

  @Test
  public void shouldHaveCompatibleEqualsAndHashcode() {
    final HashSet<Character> compatibleSet = new HashSet<>();
    final long seed = System.nanoTime();
    final Random r = new Random(seed);
    for (int i = 0; i < 1024; ++i) { // [N]
      final char value = (char)r.nextInt();
      compatibleSet.add(value);
      testSet.add(value);
    }

    if (r.nextBoolean()) {
      compatibleSet.add(HashCharSet.NULL);
      testSet.add(HashCharSet.NULL);
    }

    assertTrue("Fail with seed:" + seed, testSet.size() == compatibleSet.size() && testSet.containsAll(compatibleSet));
    Assert.assertEquals("Fail with seed:" + seed, compatibleSet.hashCode(), testSet.hashCode());
  }

  private static void addTwoElements(final HashCharSet obj) {
    obj.add((char)1);
    obj.add((char)101);
  }

  private static void addTwoElements(final HashSet<Character> obj) {
    obj.add((char)1);
    obj.add((char)101);
  }

  private void assertIteratorHasElements() {
    final CharIterator iterator = testSet.iterator();

    final Set<Character> values = new HashSet<>();

    assertTrue(iterator.hasNext());
    values.add(iterator.next());
    assertTrue(iterator.hasNext());
    values.add(iterator.next());
    assertFalse(iterator.hasNext());

    assertContainsElements(values);
  }

  private void assertIteratorHasElementsWithoutHasNext() {
    final CharIterator iterator = testSet.iterator();
    final Set<Character> values = new HashSet<>();

    values.add(iterator.next());
    values.add(iterator.next());

    assertContainsElements(values);
  }

  private static void assertArrayContainingElements(final Character[] result) {
    assertTrue(Arrays.asList(result).containsAll(Arrays.asList((char)1, (char)101)));
  }

  private static void assertContainsElements(final Set<Character> other) {
    assertTrue(other.containsAll(Arrays.asList((char)1, (char)101)));
  }

  private static void assertContainsElements(final CharSet other) {
    assertTrue(other.containsAll(Arrays.asList((char)1, (char)101)));
  }

  private void exhaustIterator() {
    final CharIterator iterator = testSet.iterator();
    iterator.next();
    iterator.next();
    iterator.next();
  }
}