/*
 * Decompiled with CFR 0.152.
 */
package org.jfrog.storage.util;

import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.concurrent.ExecutorService;
import javax.annotation.Nullable;
import javax.sql.DataSource;
import lombok.NonNull;
import org.apache.commons.lang.StringUtils;
import org.jfrog.storage.DbType;
import org.jfrog.storage.JFrogDataSource;
import org.jfrog.storage.JdbcHelper;
import org.jfrog.storage.QueryControls;
import org.jfrog.storage.util.DbStatementUtils;
import org.jfrog.storage.util.functional.IOSQLThrowingConsumer;
import org.jfrog.storage.util.functional.SQLThrowingConsumer;
import org.jfrog.storage.util.functional.SQLThrowingFunction;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.jdbc.datasource.DataSourceUtils;

public abstract class DbUtils {
    private static final Logger log = LoggerFactory.getLogger(DbUtils.class);
    public static final int DEFAULT_TIMEOUT_SEC = 3600;
    public static final int DEFAULT_TIMEOUT_MILLISECONDS = 3600000;
    public static final String VALUES = " VALUES";
    static final String COLUMN_SIZE = "COLUMN_SIZE";
    static final String IS_NULLABLE = "IS_NULLABLE";

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void close(@Nullable Connection con, @Nullable Statement stmt, @Nullable ResultSet rs) {
        try {
            DbUtils.close(rs);
        }
        finally {
            try {
                DbStatementUtils.close(stmt);
            }
            finally {
                DbUtils.close(con);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void close(@Nullable Connection con, @Nullable Statement stmt, @Nullable ResultSet rs, @Nullable DataSource ds) {
        try {
            DbUtils.close(rs);
        }
        finally {
            try {
                DbStatementUtils.close(stmt);
            }
            finally {
                DbUtils.close(con, ds);
            }
        }
    }

    public static void close(@Nullable Connection con) {
        DbUtils.close(con, null);
    }

    public static void close(@Nullable Connection con, @Nullable DataSource ds) {
        if (con != null) {
            try {
                DataSourceUtils.doReleaseConnection((Connection)con, (DataSource)ds);
            }
            catch (SQLException e) {
                log.trace("Could not close JDBC connection", (Throwable)e);
            }
            catch (Exception e) {
                log.trace("Unexpected exception when closing JDBC connection", (Throwable)e);
            }
        }
    }

    public static void close(@Nullable Statement stmt) {
        DbStatementUtils.close(stmt);
    }

    public static void close(@Nullable ResultSet rs) {
        if (rs != null) {
            try {
                rs.close();
            }
            catch (SQLException e) {
                log.trace("Could not close JDBC result set", (Throwable)e);
            }
            catch (Exception e) {
                log.trace("Unexpected exception when closing JDBC result set", (Throwable)e);
            }
        }
    }

    public static void executeSqlStream(Connection con, InputStream in) throws IOException, SQLException {
        DbStatementUtils.executeSqlStream(con, in);
    }

    public static void closeDataSource(DataSource dataSource) {
        if (dataSource == null) {
            return;
        }
        if (dataSource instanceof Closeable) {
            try {
                ((Closeable)((Object)dataSource)).close();
            }
            catch (Exception e) {
                String msg = "Error closing the data source " + dataSource + " due to:" + e.getMessage();
                if (log.isDebugEnabled()) {
                    log.error(msg, (Throwable)e);
                }
                log.error(msg);
            }
        }
    }

    public static String getActiveSchema(Connection conn, DbType dbType) throws SQLException {
        String schema;
        switch (dbType) {
            case MYSQL: 
            case MARIADB: 
            case POSTGRESQL: 
            case DERBY: {
                schema = conn.getSchema();
                break;
            }
            case ORACLE: {
                schema = conn.getMetaData().getUserName();
                break;
            }
            case MSSQL: {
                schema = null;
                break;
            }
            default: {
                throw new IllegalArgumentException("Unrecognized db type: " + dbType.name());
            }
        }
        if (log.isDebugEnabled()) {
            log.debug("Active schema resolved to: {}, DB type: {}", (Object)schema, (Object)dbType.name());
        }
        return schema;
    }

    public static String getActiveCatalog(Connection conn, DbType dbType) {
        String catalog = null;
        try {
            switch (dbType) {
                case DERBY: 
                case ORACLE: {
                    break;
                }
                case MYSQL: 
                case MARIADB: 
                case POSTGRESQL: 
                case MSSQL: {
                    catalog = conn.getCatalog();
                    break;
                }
                default: {
                    throw new IllegalArgumentException("Unrecognized db type: " + dbType.name());
                }
            }
        }
        catch (Exception e) {
            log.error("Can't obtain active catalog from db: ", (Throwable)e);
        }
        log.debug("Current active catalog: {}", (Object)catalog);
        return catalog;
    }

    public static String normalizedName(String name, DatabaseMetaData metaData) throws SQLException {
        if (metaData.storesLowerCaseIdentifiers()) {
            name = name.toLowerCase();
        } else if (metaData.storesUpperCaseIdentifiers()) {
            name = name.toUpperCase();
        }
        return name;
    }

    public static boolean tableExists(JdbcHelper jdbcHelper, DbType dbType, String tableName) throws SQLException {
        return DbUtils.withMetadata(jdbcHelper, metadata -> DbUtils.tableExists(metadata, dbType, tableName));
    }

    public static boolean tableExists(DatabaseMetaData metadata, DbType dbType, String tableName) throws SQLException {
        tableName = DbUtils.normalizedName(tableName, metadata);
        String activeCatalog = DbUtils.getActiveCatalog(metadata.getConnection(), dbType);
        String activeSchema = DbUtils.getActiveSchema(metadata.getConnection(), dbType);
        log.debug("Searching for table '{}' under schema '{}' catalog '{}'", new Object[]{tableName, activeSchema, activeCatalog});
        try (ResultSet rs = metadata.getTables(activeCatalog, activeSchema, tableName, new String[]{"TABLE"});){
            boolean hasNext = rs.next();
            if (hasNext) {
                String table = rs.getString(1);
                log.debug("Searched for table: '{}'. Got result: '{}'", (Object)tableName, (Object)table);
            }
            boolean bl = hasNext;
            return bl;
        }
    }

    public static boolean columnExists(JdbcHelper jdbcHelper, DbType dbType, String tableName, String columnName) throws SQLException {
        return DbUtils.withMetadata(jdbcHelper, metadata -> DbUtils.columnExists(metadata, dbType, tableName, columnName));
    }

    public static boolean columnExists(DatabaseMetaData metadata, DbType dbType, String tableName, String columnName) throws SQLException {
        columnName = DbUtils.normalizedName(columnName, metadata);
        tableName = DbUtils.normalizedName(tableName, metadata);
        String activeSchema = DbUtils.getActiveSchema(metadata.getConnection(), dbType);
        String activeCatalog = DbUtils.getActiveCatalog(metadata.getConnection(), dbType);
        log.debug("Searching for column '{}' in table '{}' under schema '{}' catalog '{}'", new Object[]{columnName, tableName, activeSchema, activeCatalog});
        try (ResultSet rs = metadata.getColumns(activeCatalog, activeSchema, tableName, columnName);){
            boolean found = rs.next();
            log.debug("column '{}' in table '{}' was found: {}", new Object[]{columnName, tableName, found});
            boolean bl = found;
            return bl;
        }
    }

    public static int getColumnSize(JdbcHelper jdbcHelper, DbType dbType, String tableName, String columnName) throws SQLException {
        return DbUtils.withMetadata(jdbcHelper, metadata -> DbUtils.getColumnSize(metadata, dbType, tableName, columnName));
    }

    public static int getColumnSize(DatabaseMetaData metadata, DbType dbType, String tableName, String columnName) throws SQLException {
        return DbUtils.withColumnMD(metadata, dbType, tableName, columnName, rs -> rs.getInt(COLUMN_SIZE));
    }

    public static boolean isColumnNullable(DatabaseMetaData metadata, DbType dbType, String tableName, String columnName) throws SQLException {
        return DbUtils.withColumnMD(metadata, dbType, tableName, columnName, rs -> {
            String isNullableNormalized = DbUtils.normalizedName(IS_NULLABLE, metadata);
            String nullable = rs.getString(isNullableNormalized);
            log.debug("column '{}' table '{}' dbType '{}' isNullable='{}'", new Object[]{columnName, tableName, dbType, nullable});
            return "YES".equals(nullable);
        });
    }

    private static <T> T withColumnMD(DatabaseMetaData metadata, DbType dbType, String tableName, String columnName, SQLThrowingFunction<ResultSet, T, SQLException> whatToDo) throws SQLException {
        String normalizedColumnName = DbUtils.normalizedName(columnName, metadata);
        String normalizedTableName = DbUtils.normalizedName(tableName, metadata);
        String activeSchema = DbUtils.getActiveSchema(metadata.getConnection(), dbType);
        String activeCatalog = DbUtils.getActiveCatalog(metadata.getConnection(), dbType);
        log.debug("Searching for column '{}' in table '{}' under schema '{}' catalog '{}'", new Object[]{normalizedColumnName, normalizedTableName, activeSchema, activeCatalog});
        try (ResultSet column = metadata.getColumns(activeCatalog, activeSchema, normalizedTableName, normalizedColumnName);){
            if (column.next()) {
                log.debug("column '{}' in table '{}' was found", (Object)normalizedColumnName, (Object)normalizedTableName);
                T t = whatToDo.apply(column);
                return t;
            }
        }
        throw new RuntimeException("Column: " + normalizedColumnName + " was not found on table: " + normalizedTableName);
    }

    public static boolean indexExists(JdbcHelper jdbcHelper, String tableName, @Nullable String columnName, String indexName, DbType dbType) throws SQLException {
        return DbUtils.withConnection(jdbcHelper, conn -> DbUtils.indexExists(jdbcHelper, conn, dbType, tableName, columnName, indexName, false));
    }

    public static boolean indexExists(JdbcHelper jdbcHelper, Connection conn, DbType dbType, String tableName, @Nullable String columnName, String indexName, boolean strictDerby) throws SQLException {
        DatabaseMetaData metadata = conn.getMetaData();
        String normalizedTableName = DbUtils.normalizedName(tableName, metadata);
        String normalizedColName = StringUtils.isBlank((String)columnName) ? columnName : DbUtils.normalizedName(columnName, metadata);
        String normalizedIndexName = DbUtils.normalizedName(indexName, metadata);
        if (!DbUtils.tableExists(metadata, dbType, tableName) || !DbUtils.columnExists(metadata, dbType, tableName, columnName)) {
            return false;
        }
        String activeCatalog = DbUtils.getActiveCatalog(conn, dbType);
        String activeSchema = DbUtils.getActiveSchema(conn, dbType);
        if (DbType.DERBY.equals((Object)dbType)) {
            return DbUtils.indexExistsDerby(metadata, activeCatalog, activeSchema, normalizedTableName, normalizedColName, normalizedIndexName, strictDerby);
        }
        if (DbType.ORACLE.equals((Object)dbType)) {
            return DbUtils.indexExistsOracle(jdbcHelper, activeSchema, normalizedTableName, normalizedColName, normalizedIndexName);
        }
        return DbUtils.indexExists(metadata, activeCatalog, activeSchema, normalizedTableName, normalizedColName, normalizedIndexName);
    }

    private static boolean indexExistsOracle(JdbcHelper jdbcHelper, String activeSchema, String normalizedTableName, String normalizedColName, String normalizedIndexName) throws SQLException {
        int i = jdbcHelper.executeSelectCount("SELECT COUNT(*) FROM ALL_IND_COLUMNS WHERE INDEX_NAME = ? AND COLUMN_NAME = ? AND TABLE_NAME = ? AND INDEX_OWNER = ? ", normalizedIndexName, normalizedColName, normalizedTableName, activeSchema);
        return i > 0;
    }

    private static boolean indexExists(DatabaseMetaData metadata, String activeCatalog, String activeSchema, String tableName, @Nullable String columnName, String indexName) throws SQLException {
        String colNameColumn = DbUtils.normalizedName("COLUMN_NAME", metadata);
        String idxNameColumn = DbUtils.normalizedName("INDEX_NAME", metadata);
        String tblNameColumn = DbUtils.normalizedName("TABLE_NAME", metadata);
        try (ResultSet rs = metadata.getIndexInfo(activeCatalog, activeSchema, tableName, false, false);){
            boolean bl = DbUtils.indexExists(tableName, columnName, indexName, colNameColumn, idxNameColumn, tblNameColumn, rs);
            return bl;
        }
    }

    private static boolean indexExists(String tableName, @Nullable String columnName, String indexName, String colNameColumn, String idxNameColumn, String tblNameColumn, ResultSet rs) throws SQLException {
        while (rs.next()) {
            if (!indexName.equals(rs.getString(idxNameColumn)) || !tableName.equals(rs.getString(tblNameColumn))) continue;
            if (StringUtils.isNotBlank((String)columnName)) {
                if (!columnName.equals(rs.getString(colNameColumn))) continue;
                return true;
            }
            return true;
        }
        return false;
    }

    private static boolean indexExistsDerby(DatabaseMetaData metadata, String activeCatalog, String activeSchema, String tableName, String columnName, String indexName, boolean strict) throws SQLException {
        if (DbUtils.indexExists(metadata, activeCatalog, activeSchema, tableName, columnName, indexName)) {
            return true;
        }
        if (!strict) {
            if (StringUtils.isBlank((String)columnName)) {
                log.warn("Can't search for an index on Derby without column name");
                return false;
            }
            return DbUtils.indexExistsDerby(metadata, activeCatalog, activeSchema, tableName, columnName);
        }
        return false;
    }

    private static boolean indexExistsDerby(DatabaseMetaData metadata, String activeCatalog, String activeSchema, String tableName, String columnName) throws SQLException {
        String colNameColumn = DbUtils.normalizedName("COLUMN_NAME", metadata);
        try (ResultSet rs = metadata.getIndexInfo(activeCatalog, activeSchema, tableName, false, false);){
            while (rs.next()) {
                if (!rs.getString(colNameColumn).equals(columnName)) continue;
                boolean bl = true;
                return bl;
            }
        }
        return false;
    }

    public static boolean foreignKeyExists(JdbcHelper jdbcHelper, DbType dbType, String tableName, String foreignKeyName) throws SQLException {
        return DbUtils.withConnection(jdbcHelper, conn -> DbUtils.foreignKeyExists(conn, dbType, tableName, foreignKeyName));
    }

    private static boolean foreignKeyExists(Connection conn, DbType dbType, String tableName, String foreignKeyName) throws SQLException {
        DatabaseMetaData metadata = conn.getMetaData();
        tableName = DbUtils.normalizedName(tableName, metadata);
        foreignKeyName = DbUtils.normalizedName(foreignKeyName, metadata);
        if (StringUtils.isBlank((String)tableName) || StringUtils.isBlank((String)foreignKeyName)) {
            throw new IllegalStateException("Could not resolve db-specific identifier names");
        }
        if (!DbUtils.tableExists(metadata, dbType, tableName)) {
            return false;
        }
        return DbUtils.foreignKeyExists(metadata, DbUtils.getActiveCatalog(conn, dbType), DbUtils.getActiveSchema(conn, dbType), tableName, foreignKeyName);
    }

    private static boolean foreignKeyExists(DatabaseMetaData metadata, String activeCatalog, String activeSchema, String tableName, String keyName) throws SQLException {
        String fkNameColumn = DbUtils.normalizedName("FK_NAME", metadata);
        String fkTableColumn = DbUtils.normalizedName("FKTABLE_NAME", metadata);
        try (ResultSet rs = metadata.getImportedKeys(activeCatalog, activeSchema, tableName);){
            while (rs.next()) {
                if (!rs.getString(fkNameColumn).equals(keyName) || !rs.getString(fkTableColumn).equals(tableName)) continue;
                boolean bl = true;
                return bl;
            }
        }
        return false;
    }

    public static String getDbTypeNameForSqlResources(DbType dbType) {
        String dbTypeName = dbType.name();
        if (dbTypeName.equals(DbType.MARIADB.name())) {
            log.debug("Identified MariaDB as DB type, using MySql conversion files");
            return DbType.MYSQL.name().toLowerCase();
        }
        return dbTypeName.toLowerCase();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void doWithConnection(JdbcHelper jdbcHelper, SQLThrowingConsumer<Connection, SQLException> whatToDo) throws SQLException {
        JFrogDataSource ds = null;
        Connection conn = null;
        try {
            ds = jdbcHelper.getDataSource();
            conn = ds.getConnection();
            whatToDo.accept(conn);
        }
        catch (Throwable throwable) {
            DbUtils.close(conn, ds);
            throw throwable;
        }
        DbUtils.close(conn, ds);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void doStreamWithConnection(JdbcHelper jdbcHelper, IOSQLThrowingConsumer<Connection, SQLException, IOException> whatToDo) throws IOException, SQLException {
        JFrogDataSource ds = null;
        Connection conn = null;
        try {
            ds = jdbcHelper.getDataSource();
            conn = ds.getConnection();
            whatToDo.accept(conn);
        }
        catch (Throwable throwable) {
            DbUtils.close(conn, ds);
            throw throwable;
        }
        DbUtils.close(conn, ds);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static <T> T withConnection(JdbcHelper jdbcHelper, SQLThrowingFunction<Connection, T, SQLException> whatToDo) throws SQLException {
        T t;
        JFrogDataSource ds = null;
        Connection conn = null;
        try {
            ds = jdbcHelper.getDataSource();
            conn = ds.getConnection();
            t = whatToDo.apply(conn);
        }
        catch (Throwable throwable) {
            DbUtils.close(conn, ds);
            throw throwable;
        }
        DbUtils.close(conn, ds);
        return t;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static <T> T withMetadata(JdbcHelper jdbcHelper, SQLThrowingFunction<DatabaseMetaData, T, SQLException> whatToDo) throws SQLException {
        T t;
        JFrogDataSource ds = null;
        Connection conn = null;
        try {
            ds = jdbcHelper.getDataSource();
            conn = ds.getConnection();
            t = whatToDo.apply(conn.getMetaData());
        }
        catch (Throwable throwable) {
            DbUtils.close(conn, ds);
            throw throwable;
        }
        DbUtils.close(conn, ds);
        return t;
    }

    public static String printResultSet(ResultSet resultSet) {
        StringBuilder resultBuilder = new StringBuilder();
        try {
            int i;
            ResultSetMetaData rsmd = resultSet.getMetaData();
            int columnsNumber = rsmd.getColumnCount();
            for (i = 1; i <= columnsNumber; ++i) {
                resultBuilder.append(rsmd.getColumnName(i)).append(",");
            }
            resultBuilder.append("\n");
            while (resultSet.next()) {
                for (i = 1; i <= columnsNumber; ++i) {
                    resultBuilder.append(resultSet.getString(i)).append(",");
                }
                resultBuilder.append("\n");
            }
            resultBuilder.append("\n");
        }
        catch (Exception e) {
            log.error("", (Throwable)e);
        }
        return resultBuilder.toString();
    }

    public static Connection getConnection(QueryControls controls, DataSource dataSource, ExecutorService executorService) throws SQLException {
        Connection connection = DataSourceUtils.doGetConnection((DataSource)dataSource);
        DbUtils.setNetworkTimeout(connection, controls, executorService);
        return connection;
    }

    private static void setNetworkTimeout(Connection con, QueryControls controls, ExecutorService executorService) throws SQLException {
        if (controls.getNetworkTimeout() > 0) {
            log.debug("Setting the network timeout to be {} sec", (Object)controls.getNetworkTimeout());
            con.setNetworkTimeout(executorService, controls.getNetworkTimeout() * 1000);
        }
    }

    public static boolean dropTableIfExists(JdbcHelper jdbcHelper, DbType dbType, @NonNull String table) throws SQLException {
        if (table == null) {
            throw new NullPointerException("table is marked non-null but is null");
        }
        if (DbUtils.tableExists(jdbcHelper, dbType, table)) {
            return DbUtils.withConnection(jdbcHelper, conn -> conn.createStatement().execute("DROP TABLE " + table));
        }
        return false;
    }

    private DbUtils() {
    }
}

