/*
 * Decompiled with CFR 0.152.
 */
package org.jfrog.storage;

import com.google.common.collect.Maps;
import java.io.Closeable;
import java.io.IOException;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.Map;
import java.util.Optional;
import javax.annotation.Nonnull;
import org.apache.commons.lang.StringUtils;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.jfrog.storage.DatabaseHelper;
import org.jfrog.storage.DbType;
import org.jfrog.storage.JFrogDataSource;
import org.jfrog.storage.QueryControls;
import org.jfrog.storage.dbtype.DbSpecificHelper;
import org.jfrog.storage.dbtype.DefaultDbTypeHelper;
import org.jfrog.storage.dbtype.OracleSpecificHelper;
import org.jfrog.storage.util.DbStatementUtils;
import org.jfrog.storage.util.DbUtils;
import org.jfrog.storage.util.PerfTimer;
import org.jfrog.storage.util.SqlTracer;
import org.jfrog.storage.util.TxHelper;
import org.jfrog.storage.wrapper.BlobWrapper;
import org.jfrog.storage.wrapper.ResultSetWrapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class JdbcHelper
implements Closeable,
DatabaseHelper {
    private static final Logger log = LoggerFactory.getLogger(JdbcHelper.class);
    public static final int NO_DB_ID = -1;
    private final JFrogDataSource dataSource;
    private boolean closed = false;
    private final SqlTracer tracer = new SqlTracer();
    private final Map<DbType, DbSpecificHelper> dbSpecificHelpers = Maps.newHashMap();

    public JdbcHelper(JFrogDataSource dataSource) {
        this.dataSource = dataSource;
        this.initDbSpecificTypeHelpers();
    }

    private Connection getConnection(QueryControls controls) throws SQLException {
        this.assertNotClosed();
        return DbUtils.getConnection(controls, this.dataSource, this.dataSource.getExecutorService());
    }

    private void initDbSpecificTypeHelpers() {
        this.dbSpecificHelpers.put(DbType.ORACLE, new OracleSpecificHelper());
        this.dbSpecificHelpers.put(null, new DefaultDbTypeHelper());
    }

    public JFrogDataSource getDataSource() {
        this.assertNotClosed();
        return this.dataSource;
    }

    @Override
    @Nonnull
    public ResultSet executeSelect(String query, Object ... params) throws SQLException {
        return this.executeSelect(QueryControls.DEFAULT_QUERY_CONTROLS, query, params);
    }

    @Nonnull
    public ResultSet executeSelect(QueryControls controls, String query, Object ... params) throws SQLException {
        if (this.closed) {
            throw new IllegalStateException("DataSource is closed cannot execute select query:\n'" + query + "'");
        }
        PerfTimer timer = this.traceSqlStart(query, params);
        Connection con = null;
        Statement stmt = null;
        ResultSet rs = null;
        try {
            con = this.getConnection(controls);
            this.allowDirtyReads(controls.isAllowDirtyReads(), con);
            if (params == null || params.length == 0) {
                stmt = con.createStatement();
                this.setQueryTimeout(stmt, controls);
                rs = stmt.executeQuery(query);
            } else {
                PreparedStatement pstmt = con.prepareStatement(DbStatementUtils.parseInListQuery(query, params));
                stmt = pstmt;
                this.setQueryTimeout(stmt, controls);
                DbStatementUtils.setParamsToStmt(pstmt, params);
                rs = pstmt.executeQuery();
            }
            if (!TxHelper.isInTransaction() && !con.getAutoCommit()) {
                con.commit();
            }
            this.traceSqlFinish(query, params, timer);
            return ResultSetWrapper.newInstance(con, stmt, rs, this.dataSource, controls.isAllowDirtyReads());
        }
        catch (Exception e) {
            DbUtils.close(con, stmt, rs, this.dataSource);
            if (e instanceof SQLException) {
                throw (SQLException)e;
            }
            throw new SQLException("Unexpected exception: " + e.getMessage(), e);
        }
    }

    @Override
    public int executeUpdate(String query, Object ... params) throws SQLException {
        return this.executeUpdate(QueryControls.DEFAULT_QUERY_CONTROLS, query, params);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public int executeUpdate(QueryControls controls, String query, Object ... params) throws SQLException {
        int n;
        if (this.closed) {
            throw new IllegalStateException("DataSource is closed cannot execute update query:\n'" + query + "'");
        }
        PerfTimer timer = this.traceSqlStart(query, params);
        Connection con = null;
        Statement stmt = null;
        try {
            int results;
            con = this.getConnection(controls);
            if (params == null || params.length == 0) {
                stmt = con.createStatement();
                this.setQueryTimeout(stmt, controls);
                results = stmt.executeUpdate(query);
            } else {
                PreparedStatement pstmt = con.prepareStatement(DbStatementUtils.parseInListQuery(query, params));
                stmt = pstmt;
                this.setQueryTimeout(stmt, controls);
                DbStatementUtils.setParamsToStmt(pstmt, params);
                results = pstmt.executeUpdate();
            }
            this.traceSqlFinish(query, params, timer, results);
            n = results;
        }
        catch (Throwable throwable) {
            DbUtils.close(con, stmt, null, this.dataSource);
            throw throwable;
        }
        DbUtils.close(con, stmt, null, this.dataSource);
        return n;
    }

    public int executeSelectCount(String query, Object ... params) throws SQLException {
        try (ResultSet resultSet = this.executeSelect(query, params);){
            int count = 0;
            if (resultSet.next()) {
                count = resultSet.getInt(1);
            }
            int n = count;
            return n;
        }
    }

    @Override
    public long executeSelectLong(String query, Object ... params) throws SQLException {
        try (ResultSet resultSet = this.executeSelect(query, params);){
            long result = -1L;
            if (resultSet.next()) {
                result = resultSet.getLong(1);
            }
            long l = result;
            return l;
        }
    }

    public static String resolveQuery(String sql, Object[] params) {
        int paramsLength;
        int expectedParamsCount = StringUtils.countMatches((String)sql, (String)"?");
        int n = paramsLength = params == null ? 0 : params.length;
        if (expectedParamsCount != paramsLength) {
            log.warn("Unexpected parameters count. Expected {} but got {}", (Object)expectedParamsCount, (Object)paramsLength);
        }
        if (params == null || params.length == 0) {
            return sql;
        }
        sql = expectedParamsCount == paramsLength ? JdbcHelper.buildSqlWithParams(sql, params) : JdbcHelper.buildErrorSql(sql, params);
        return sql;
    }

    @NotNull
    private static String buildErrorSql(String sql, Object[] params) {
        StringBuilder builder = new StringBuilder();
        builder.append("Executing SQL: '").append(sql).append("'");
        builder.append(" with params: ");
        for (int i = 0; i < params.length; ++i) {
            builder.append("'");
            builder.append(params[i]);
            builder.append("'");
            if (i >= params.length - 1) continue;
            builder.append(", ");
        }
        sql = builder.toString();
        return sql;
    }

    private static String buildSqlWithParams(String sql, Object[] params) {
        Object[] printParams = JdbcHelper.buildPrintParams(params);
        sql = sql.replace("%", "%%").replaceAll("\\?", "%s");
        sql = String.format(sql, printParams);
        return sql;
    }

    @NotNull
    private static Object[] buildPrintParams(Object[] params) {
        Object[] printParams = new Object[params.length];
        for (int i = 0; i < params.length; ++i) {
            Object current = params[i];
            if (current == null) {
                current = "NULL";
            } else if (current instanceof String) {
                current = "'" + params[i] + "'";
            } else if (current instanceof BlobWrapper) {
                current = "BLOB(length: " + ((BlobWrapper)params[i]).getLength() + ")";
            }
            printParams[i] = current;
        }
        return printParams;
    }

    public boolean isClosed() {
        return this.closed;
    }

    public long getSelectQueriesCount() {
        return this.tracer.getSelectQueriesCount();
    }

    public long getUpdateQueriesCount() {
        return this.tracer.getUpdateQueriesCount();
    }

    public SqlTracer getTracer() {
        return this.tracer;
    }

    DbSpecificHelper getDbSpecificHelper(DbType dbType) {
        return Optional.ofNullable(this.dbSpecificHelpers.get((Object)dbType)).orElseGet(() -> this.dbSpecificHelpers.get(null));
    }

    @Override
    public void close() throws IOException {
        if (!this.isClosed()) {
            this.destroy();
        }
    }

    public void destroy() {
        this.closed = true;
        DbUtils.closeDataSource(this.dataSource);
    }

    private void assertNotClosed() {
        if (this.closed) {
            throw new IllegalStateException("DataSource is closed");
        }
    }

    private void setQueryTimeout(Statement stmt, QueryControls controls) throws SQLException {
        if (controls.getQueryTimeout() > 0) {
            stmt.setQueryTimeout(controls.getQueryTimeout());
        }
    }

    private void allowDirtyReads(boolean allowDirtyReads, Connection con) throws SQLException {
        if (allowDirtyReads) {
            con.setTransactionIsolation(1);
        }
    }

    private PerfTimer traceSqlStart(String query, Object[] params) {
        this.tracer.traceSelectQuery(query);
        if (log.isDebugEnabled() && this.includeQueryLogging(query)) {
            String resolved = JdbcHelper.resolveQuery(query, params);
            log.trace("Executing SQL: '{}'.", (Object)resolved);
            return this.initPerfTimer();
        }
        return null;
    }

    private void traceSqlFinish(String query, Object[] params, PerfTimer timer) {
        this.traceSqlFinish(query, params, timer, -1);
    }

    private void traceSqlFinish(String query, Object[] params, PerfTimer timer, int results) {
        if (timer != null && log.isDebugEnabled()) {
            timer.stop();
            if (results < 0) {
                log.debug("Query returned in {}: \"{}\"", (Object)timer, (Object)JdbcHelper.resolveQuery(query, params));
            } else {
                log.debug("Query returned in {} with {} result{}: \"{}\"", new Object[]{timer, results, results > 1 ? "s" : "", JdbcHelper.resolveQuery(query, params)});
            }
        }
    }

    private boolean includeQueryLogging(String query) {
        return log.isTraceEnabled() || !this.excludeQueryLogging(query);
    }

    protected boolean excludeQueryLogging(String query) {
        return false;
    }

    @Nullable
    private PerfTimer initPerfTimer() {
        return log.isDebugEnabled() ? new PerfTimer() : null;
    }
}

