package org.jfrog.storage.wrapper;

import org.jfrog.storage.DbProperties;
import org.jfrog.storage.DbType;
import org.jfrog.storage.JFrogDataSource;

import javax.sql.DataSource;
import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.sql.Connection;
import java.sql.SQLException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import static org.jfrog.storage.util.DbUtils.DEFAULT_TIMEOUT_MILLISECONDS;

/**
 * A dynamic proxy to wrap {@link org.jfrog.storage.JFrogDataSource}.
 * Its purpose is to manage the data sources and to have the common methods in one place.
 *
 * @author Alexei Vainshtein
 */
public class JFrogDataSourceWrapper implements InvocationHandler {

    private ExecutorService executorService;
    private DataSource dataSource;
    private DbType dbType;
    private boolean networkTimeoutEnabled;

    private JFrogDataSourceWrapper(DataSource dataSource, DbProperties dbProps) {
        this.dataSource = dataSource;
        this.dbType = dbProps.getDbType();
        this.networkTimeoutEnabled = dbProps.getBooleanProperty("networkTimeoutEnabled", true);
        if (networkTimeoutEnabled) {
            this.executorService =
                    new ThreadPoolExecutor(0, dbProps.getMaxActiveConnections(), 0, TimeUnit.MILLISECONDS,
                            new LinkedBlockingQueue<>());
        }
    }

    /**
     * Creates a new {@link JFrogConnectionWrapper} with info regarding the database.
     *
     * @param dataSource - The associated {@link JFrogDataSource}
     * @param dbProps    - The database configuration info {@link DbProperties}
     * @return Proxy to the JFrog Data Source
     */
    public static JFrogDataSource newInstance(JFrogDataSource dataSource, DbProperties dbProps) {
        JFrogDataSourceWrapper proxy = new JFrogDataSourceWrapper(dataSource, dbProps);
        return (JFrogDataSource) Proxy.newProxyInstance(dataSource.getClass().getClassLoader(),
                new Class<?>[] {JFrogDataSource.class}, proxy);
    }

    /**
     * @see InvocationHandler#invoke(Object, Method, Object[])
     */
    @Override
    public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {
        if ("getConnection".equals(method.getName())) {
            return getConnection();
        }
        if ("getExecutorService".equalsIgnoreCase(method.getName())) {
            return executorService;
        }
        if ("unwrap".equalsIgnoreCase(method.getName())) {
            return unwrap((Class) args[0]);
        }
        if ("isWrapperFor".equalsIgnoreCase(method.getName())) {
            return isWrapperFor(((Class) args[0]));
        }
        if ("close".equalsIgnoreCase(method.getName())) {
            method.invoke(dataSource, args);
            close();
            return null;
        }

        return method.invoke(dataSource, args);
    }

    /**
     * Creates JFrog proxy connection and setting the timeout to the {@link DEFAULT_TIMEOUT_MILLISECONDS}
     *
     * @return the connection with default timeout set.
     */
    private Connection getConnection() throws SQLException {
        Connection conn = dataSource.getConnection();
        Connection connection = JFrogConnectionWrapper.newInstance(conn, dbType, networkTimeoutEnabled);
        connection.setNetworkTimeout(executorService, DEFAULT_TIMEOUT_MILLISECONDS);
        return connection;
    }

    /**
     * Closes the executor service if it was init
     */
    private void close() {
        if (executorService != null) {
            executorService.shutdown();
        }
    }

    /**
     * @see java.sql.Wrapper#unwrap(Class)
     */
    private <T> T unwrap(Class<T> aClass) throws SQLException {
        if (aClass.isInstance(dataSource)) {
            return aClass.cast(dataSource);
        }
        throw new SQLException("DataSource of type [" + dataSource.getClass().getName() +
                "] cannot be unwrapped as [" + aClass.getName() + "]");
    }

    /**
     * @see java.sql.Wrapper#isWrapperFor(Class)
     */
    private boolean isWrapperFor(Class<?> aClass) {
        return aClass.isInstance(dataSource);
    }
}
