package org.jfrog.storage.wrapper;

import org.jfrog.storage.DbType;

import java.io.IOException;
import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.sql.Connection;
import java.sql.SQLException;
import java.util.concurrent.Executor;

/**
 * A dynamic proxy to wrap {@link Connection}.
 * Its purpose is to have the ability to manage the network timeout per different configurations.
 *
 * @author Alexei Vainshtein
 */
public class JFrogConnectionWrapper implements InvocationHandler {

    private Connection connection;
    private DbType dbType;
    private boolean networkTimeoutEnabled;
    private int networkTimeoutConfigured;

    private JFrogConnectionWrapper(Connection connection, DbType dbType, boolean networkTimeoutEnabled) {
        this.connection = connection;
        this.dbType = dbType;
        this.networkTimeoutEnabled = networkTimeoutEnabled;
    }

    /**
     * Creates a new {@link Connection} with the ability to manage the network timeout.
     *
     * @param con                   - The associated {@link Connection}
     * @param dbType                - The database type that is used
     * @param networkTimeoutEnabled - If the ability to set network timeout is enabled. Default true.
     * @return Proxy to the connection
     */
    public static Connection newInstance(Connection con, DbType dbType, boolean networkTimeoutEnabled) {
        JFrogConnectionWrapper proxy = new JFrogConnectionWrapper(con, dbType, networkTimeoutEnabled);
        return (Connection) Proxy.newProxyInstance(con.getClass().getClassLoader(),
                new Class<?>[] {Connection.class}, proxy);
    }

    /**
     * @see Connection#setNetworkTimeout(Executor, int)
     * Not all databases or the database drivers supoprting this.
     * Derby is not supporting this feature.
     * Also, Oracle driver supporting this feature from version 12 and ojdbc8.
     */
    private void setNetworkTimeout(Executor executor, int milliseconds) throws SQLException {
        if (!networkTimeoutEnabled || this.dbType == DbType.DERBY || networkTimeoutConfigured == milliseconds) {
            return;
        }

        if (isUnsupportedOracleDriver() || isUnsupportedMSSQLDriver()) {
            return;
        }
        connection.setNetworkTimeout(executor, milliseconds);
        networkTimeoutConfigured = milliseconds;
    }

    /**
     * @see Connection#getNetworkTimeout()
     * Not all databases or the database drivers supoprting this.
     * Derby is not supporting this feature.
     * Also, Oracle driver supporting this feature from version 12 and ojdbc8.
     */
    private int getNetworkTimeout() throws SQLException {
        if (!networkTimeoutEnabled || this.dbType == DbType.DERBY) {
            return 0;
        }

        if (isUnsupportedOracleDriver() || isUnsupportedMSSQLDriver()) {
            return 0;
        }
        return connection.getNetworkTimeout();
    }

    private boolean isUnsupportedOracleDriver() throws SQLException {
        // Oracle driver supports get and set for network only from version 12 and above that compiled with jdk8
        return this.dbType == DbType.ORACLE && connection.getMetaData().getDriverMajorVersion() <= 11;
    }

    private boolean isUnsupportedMSSQLDriver() throws SQLException {
        // MSSQL driver supports get and set for network only from version 6 and above
        return this.dbType == DbType.MSSQL && connection.getMetaData().getDriverMajorVersion() < 6;
    }

    /**
     * @see InvocationHandler#invoke(Object, Method, Object[])
     */
    @Override
    public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {
        if ("setNetworkTimeout".equals(method.getName())) {
            if (args.length != 2) {
                throw new IOException("Expected 2 arguments, however, got: " + args.length);
            }
            setNetworkTimeout((Executor) args[0], (Integer) args[1]);
            return null;
        }
        if ("getNetworkTimeout".equals(method.getName())) {
            return getNetworkTimeout();
        }
        return method.invoke(connection, args);
    }
}
