/*
 *
 * Copyright 2019 JFrog Ltd. All rights reserved.
 * JFROG PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */

package org.jfrog.storage.util;

import lombok.extern.slf4j.Slf4j;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.jfrog.storage.dbtype.DbSpecificHelper;
import org.jfrog.storage.dbtype.DefaultDbTypeHelper;
import org.jfrog.storage.dbtype.OracleSpecificHelper;
import org.jfrog.storage.wrapper.BlobWrapper;

import javax.annotation.Nullable;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

/**
 * @author Dan Feldman
 */
@Slf4j
public class DbStatementUtils {

    private DbStatementUtils() {

    }

    public static void executeSqlStream(Connection con, InputStream in) throws IOException, SQLException {
        BufferedReader reader = new BufferedReader(new InputStreamReader(in, StandardCharsets.UTF_8));
        Statement stmt = con.createStatement();
        try {
            String dbName = con.getMetaData().getDatabaseProductName();
            DbSpecificHelper dbSpecificHelper =
                    "Oracle".equals(dbName) ? new OracleSpecificHelper() : new DefaultDbTypeHelper();

            StringBuilder sb = new StringBuilder();
            String line;
            while ((line = reader.readLine()) != null) {
                if (StringUtils.isNotBlank(line) && !line.startsWith("--") && !line.startsWith("#")) {
                    sb.append(line);
                    if (line.endsWith(";")) {
                        String query = sb.substring(0, sb.length() - 1);
                        try {
                            dbSpecificHelper.executeUpdateQuery(query, stmt);
                            if (!con.getAutoCommit()) {
                                con.commit();
                            }
                        } catch (SQLException e) {
                            log.warn("Failed to execute query: {}:\n{}", e.getMessage(), query);
                            throw e;
                        }
                        sb = new StringBuilder();
                    } else {
                        sb.append("\n");
                    }
                }
            }
        } finally {
            IOUtils.closeQuietly(reader);
            close(stmt);
        }
    }

    public static String parseInListQuery(String sql, Object... params) {
        int idx = sql.indexOf("(#)");
        if (idx != -1) {
            List<Integer> iterableSizes = new ArrayList<>(1);
            for (Object param : params) {
                if (param instanceof Collection) {
                    int size = ((Collection) param).size();
                    if (size > 0) {
                        iterableSizes.add(size);
                    }
                }
            }
            if (iterableSizes.isEmpty()) {
                throw new IllegalArgumentException("Could not find collection in parameters needed for query " + sql);
            }

            StringBuilder builder = new StringBuilder(sql.substring(0, idx + 1));
            for (int i = 0; i < iterableSizes.get(0); i++) {
                if (i != 0) {
                    builder.append(',');
                }
                builder.append('?');
            }
            builder.append(sql.substring(idx + 2));
            return builder.toString();
        }
        return sql;
    }

    public static void setParamsToStmt(PreparedStatement pstmt, Object[] params) throws SQLException {
        int i = 1;
        for (Object param : params) {
            if (param instanceof Iterable) {
                for (Object p : (Iterable) param) {
                    pstmt.setObject(i++, p);
                }
            } else if (param instanceof BlobWrapper) {
                BlobWrapper blobWrapper = (BlobWrapper) param;
                if (blobWrapper.getLength() < 0) {
                    pstmt.setBinaryStream(i++, blobWrapper.getInputStream());
                } else {
                    pstmt.setBinaryStream(i++, blobWrapper.getInputStream(), blobWrapper.getLength());
                }
            } else {
                pstmt.setObject(i++, param);
            }
        }
    }

    /**
     * Closes the given statement and just logs any exception.
     *
     * @param stmt The {@link Statement} to close.
     */
    public static void close(@Nullable Statement stmt) {
        if (stmt != null) {
            try {
                stmt.close();
            } catch (SQLException e) {
                log.trace("Could not close JDBC statement", e);
            } catch (Exception e) {
                log.trace("Unexpected exception when closing JDBC statement", e);
            }
        }
    }
}
