package org.jfrog.storage.id;

import org.jfrog.storage.JFrogDataSource;
import org.jfrog.storage.StorageException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.sql.SQLException;
import java.util.concurrent.atomic.AtomicLong;

/**
 * @author Yinon Avraham.
 */
public class SimpleIdGenerator extends IdGenerator {

    private static final Logger log = LoggerFactory.getLogger(SimpleIdGenerator.class);

    private final Object indexMonitor = new Object();
    private final AtomicLong currentIndex = new AtomicLong(NO_ID);
    private volatile long maxReservedIndex = NO_ID;

    public SimpleIdGenerator(JFrogDataSource dataSource, long step, String tableName, String indexType) {
        super(dataSource, step, tableName, indexType);
    }

    @Override
    protected void initializeIndex(long currentValue) {
        maxReservedIndex = currentValue;
        if (currentValue != NO_ID) {
            currentValue++;
        }
        currentIndex.set(currentValue);
        log.debug("Initialized current index to " + currentIndex.get());
    }

    @Override
    public long nextId() {
        long value = currentIndex.getAndIncrement();
        if (isNotInRange(value)) {
            synchronized (indexMonitor) {
                value = getGoodValue(value);
            }
        }
        return value;
    }

    private long getGoodValue(long value) {
        if (isNotInRange(value)) {
            simpleUpdateIndex(value);
            if (value == NO_ID) {
                value = currentIndex.getAndIncrement();
            }
        }
        return value;
    }

    private boolean isNotInRange(long value) {
        return value > maxReservedIndex || value == NO_ID;
    }

    private void simpleUpdateIndex(long value) {
        log.trace("Starting updating index with current value={}", maxReservedIndex);
        try {
            long nextMaxCurrentIndex = Long.max(maxReservedIndex, value) + step();
            log.trace("Executing update index with value={}", nextMaxCurrentIndex);
            int rows = jdbcHelper().executeUpdate("UPDATE " + tableName() + " SET current_id = ? where index_type = ?",
                    nextMaxCurrentIndex, indexType());
            if (rows == 1) {
                maxReservedIndex = nextMaxCurrentIndex;
                log.trace("Updated index with value={}", maxReservedIndex);
            } else {
                throw new StorageException(
                        "Failed to update the unique indices table, index type " + indexType() + " does not exists!");
            }
        } catch (SQLException e) {
            throw new StorageException("Failed to update the unique indices table", e);
        }
    }

}
