package org.jfrog.storage.id;

import org.jfrog.storage.DatabaseHelper;
import org.jfrog.storage.JFrogDataSource;
import org.jfrog.storage.StorageException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.concurrent.atomic.AtomicLong;

/**
 * @author Yinon Avraham.
 */
public class RangeBasedIdGenerator extends IdGenerator {

    private static final Logger log = LoggerFactory.getLogger(RangeBasedIdGenerator.class);

    private final int maxUpdateRetries;
    private final Object rangeSync = new Object();
    private volatile IndexRange currentRange = null;

    /**
     * {@inheritDoc}
     */
    public RangeBasedIdGenerator(JFrogDataSource dataSource, long step, String tableName, String indexType,
            int maxUpdateRetries) {
        super(dataSource, step, tableName, indexType);
        this.maxUpdateRetries = maxUpdateRetries;
    }

    public RangeBasedIdGenerator(DatabaseHelper databaseHelper, long step, String tableName, String indexType,
            int maxUpdateRetries) {
        super(databaseHelper, step, tableName, indexType);
        this.maxUpdateRetries = maxUpdateRetries;
    }

    @Override
    protected void initializeIndex(long currentValue) {
        // Always set to no id to get good value on first next id call
        currentRange = new IndexRange();
    }

    @Override
    public long nextId() {
        long value = currentRange.next();
        if (value == NO_ID) {
            synchronized (rangeSync) {
                value = currentRange.next();
                if (value == NO_ID) {
                    currentRange = getNextRange();
                    value = currentRange.next();
                }
            }
        }
        return value;
    }

    private IndexRange getNextRange() {
        long maxReservedIdFromDb = currentRange.max;
        log.trace("Starting loop of updating with current max={} and {} retries", maxReservedIdFromDb,
                maxUpdateRetries);
        for (int i = 0; i < maxUpdateRetries; i++) {
            try {
                log.trace("#{} Creating select and update statements", i);
                long nextMaxCurrentIndex = maxReservedIdFromDb + step();
                log.trace("#{} executing update to next max={} from={}", i, nextMaxCurrentIndex, maxReservedIdFromDb);
                int rows = jdbcHelper().executeUpdate(
                        "UPDATE " + tableName() + " SET current_id = ? where index_type = ? AND current_id = ?",
                        nextMaxCurrentIndex, indexType(), maxReservedIdFromDb);
                if (rows == 1) {
                    if (i > 10) {
                        log.info("#{} success updating index {} with range {}-{}", i + 1, indexType(),
                                maxReservedIdFromDb, nextMaxCurrentIndex);
                    }
                    log.debug("#{} success updating index {} with range {}-{}", i + 1, indexType(),
                            maxReservedIdFromDb, nextMaxCurrentIndex);
                    return new IndexRange(nextMaxCurrentIndex);
                } else {
                    if (i > 10) {
                        log.info("#{} failed updating index {} with range {}-{}", i + 1, indexType(),
                                maxReservedIdFromDb, nextMaxCurrentIndex);
                    }
                    if (i > 5) {
                        // In PostgreSQL the select done immediately after the failing always returns an old version.
                        // TODO: Sleeping helps, should be done earlier (i==5) if an if POSTGRES can be done
                        Thread.sleep(200);
                    }
                    log.debug("#{} failed updating index {} with range {}-{}", i + 1, indexType(),
                            maxReservedIdFromDb, nextMaxCurrentIndex);
                }
                maxReservedIdFromDb = getMaxReservedIdFromDb();
            } catch (SQLException e) {
                throw new StorageException("Failed to update the unique indices table", e);
            } catch (InterruptedException e) {
                throw new StorageException("Got interrupted while updating the unique indices table", e);
            }
        }
        throw new StorageException("Failed to update the unique indices table. Too many retries.");
    }

    private long getMaxReservedIdFromDb() throws SQLException {
        try (ResultSet rs = jdbcHelper()
                .executeSelect("SELECT current_id FROM " + tableName() + " WHERE index_type = ?", indexType())) {
            return rs.next() ? rs.getLong(1) : NO_ID;
        }
    }

    class IndexRange {
        final long min, max;
        final AtomicLong current;

        IndexRange() {
            this.max = NO_ID;
            this.min = NO_ID;
            current = new AtomicLong(NO_ID);
        }

        IndexRange(long max) {
            this.max = max;
            this.min = max - step() + 1;
            current = new AtomicLong(this.min);
        }

        long next() {
            long value = current.getAndIncrement();
            if (value > max) {
                return NO_ID;
            }
            return value;
        }
    }
}
