package org.jfrog.storage.id;

import org.jfrog.storage.DatabaseHelper;
import org.jfrog.storage.JFrogDataSource;
import org.jfrog.storage.JdbcHelper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;
import java.sql.SQLException;

import static java.util.Objects.requireNonNull;
import static org.apache.commons.lang.StringUtils.isBlank;

/**
 * An abstract base implementation for an ID generator (of type long).
 * <p>
 * Requires a database table with the following schema:
 * <pre>
 * CREATE TABLE unique_ids (
 *   index_type VARCHAR(32) NOT NULL,
 *   current_id BIGINT      NOT NULL,
 *   CONSTRAINT unique_ids_pk PRIMARY KEY (index_type)
 * );
 * </pre>
 * (the table name <code>"unique_ids"</code> is only an example, the column names and types are required)
 *
 * <b>IMPORTANT:</b> The {@link #initialize()} method must be called before starting to use the generator
 * and in a single transaction
 *
 * @author Yinon Avraham.
 */
public abstract class IdGenerator {
    private static final Logger log = LoggerFactory.getLogger(IdGenerator.class);

    public static final long NO_ID = 0;

    private final DatabaseHelper databaseHelper;
    private final long step;
    private final String tableName;
    private final String indexType;

    public IdGenerator(JFrogDataSource dataSource, long step, String tableName, String indexType) {
        this(new JdbcHelper(requireNonNull(dataSource, "data source is required")), step, tableName, indexType);
    }

    public IdGenerator(DatabaseHelper databaseHelper, long step, String tableName, String indexType) {
        this.databaseHelper = requireNonNull(databaseHelper, "database helper is required");
        this.step = requirePositive(step, "step must be positive");
        this.tableName = requireNonBlank(tableName, "table name is required");
        this.indexType = requireNonBlank(indexType, "index type is required");
    }

    public long step() {
        return step;
    }

    @Nonnull
    public String tableName() {
        return tableName;
    }

    @Nonnull
    public String indexType() {
        return indexType;
    }

    @Nonnull
    protected DatabaseHelper jdbcHelper() {
        return databaseHelper;
    }

    /**
     * Initialize the generator.
     * <p>
     * <b>IMPORTANT:</b> This method must be called before starting to use the generator and in a single transaction
     * </p>
     *
     * @throws SQLException
     */
    public void initialize() throws SQLException {
        log.debug("Initializing ID generator: tableName={}, indexName={}, step={}", tableName, indexType, step);
        try {
            long initialIndex = databaseHelper
                    .executeSelectLong("SELECT current_id FROM " + tableName + " WHERE index_type = ?", indexType);
            if (initialIndex == JdbcHelper.NO_DB_ID) {
                int rows =
                        databaseHelper.executeUpdate("INSERT INTO " + tableName + " VALUES (?, ?)", indexType, NO_ID);
                if (rows == 1) {
                    log.debug("Created current unique id for the first time");
                }
                initialIndex = NO_ID;
            }
            log.debug("Initialize index to {}", initialIndex);
            initializeIndex(initialIndex);
        } catch (Exception e) {
            throw new SQLException("Could not select current index.", e);
        }
    }

    protected abstract void initializeIndex(long currentValue);

    public abstract long nextId();

    private static String requireNonBlank(String value, String message) {
        if (isBlank(value)) {
            throw new IllegalArgumentException(message + " (got: " + value + ")");
        }
        return value;
    }

    private static long requirePositive(long value, String message) {
        if (value <= 0) {
            throw new IllegalArgumentException(message + " (got: " + value + ")");
        }
        return value;
    }
}
