package org.jfrog.storage.dbtype;

import lombok.extern.slf4j.Slf4j;

import java.sql.SQLException;
import java.sql.Statement;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

/**
 * @author Noam Shemesh
 */
@Slf4j
public class OracleSpecificHelper extends DefaultDbTypeHelper {

    private static final String INSERT_INTO = "INSERT INTO ";
    private static final String VALUES = " VALUES";

    @Override
    public String getInsertMultipleValuesSql(String tableAndFields, String valuesSql, int numberOfValues) {
        return "INSERT ALL " +
                IntStream.range(0, numberOfValues)
                        .mapToObj(i -> " INTO " + tableAndFields + " VALUES " + valuesSql)
                        .collect(Collectors.joining(" ", "", " SELECT * FROM dual"));
    }

    @Override
    public void executeUpdateQuery(String query, Statement statement) throws SQLException {

        boolean insertIntoQuery = query.startsWith(INSERT_INTO) && query.contains(VALUES);
        if (insertIntoQuery) {
            for (String sql : transformInsertIntoSeparateQueries(query)) {
                log.debug("add batch query: {}", sql);
                statement.addBatch(sql);
            }
            statement.executeBatch();
        } else {
            super.executeUpdateQuery(query, statement);
        }

    }

    /**
     * transforming "INSERT INTO xxx VALUES ..." and "INSERT INTO xxx (colA,colB) VALUES ..." queries into separate
     * queries because Oracle does not support this syntax.
     * query must be in specific standard where \n separates each Values line (already done be caller to this method)
     */
    private List<String> transformInsertIntoSeparateQueries(String query) {
        int values = query.indexOf(VALUES);
        if (values == -1) {
            throw new IllegalArgumentException("Query " + query + " does not contains the keyword " + VALUES);
        }
        //baseString will be 'INSERT INTO xxx ... VALUES"
        String baseString = query.substring(0, values + VALUES.length());
        //+1 to remove the 1st \n
        String restOfString = query.substring(baseString.length() + 1);
        String[] rows = restOfString.split(",\n");

        List<String> result = Arrays.stream(rows)
                .map(row -> baseString + " " + row)
                .collect(Collectors.toList());

        log.info("Convert INSERT INTO in oracle found {} rows ", result.size());
        return result;
    }
}
