package org.jfrog.storage.dbtype;

import lombok.extern.slf4j.Slf4j;

import java.sql.SQLException;
import java.sql.Statement;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

/**
 * @author Noam Shemesh
 */
@Slf4j
public class DefaultDbTypeHelper implements DbSpecificHelper {

    @Override
    public String getInsertMultipleValuesSql(String tableAndFields, String valuesSql, int numberOfValues) {
        return "INSERT INTO " + tableAndFields + " VALUES " +
                IntStream.range(0, numberOfValues).mapToObj(i -> valuesSql).collect(
                        Collectors.joining(", "));
    }

    @Override
    public String getDeleteMultipleValuesSql(String tableAndFields, String valuesSql, int numberOfValues) {
        return "DELETE FROM " + tableAndFields + " IN (" +
                IntStream.range(0, numberOfValues).mapToObj(i -> valuesSql).collect(
                        Collectors.joining(", ")) + ")";
    }

    @Override
    public String getDeleteMultipleColumnsMultipleValuesSql(String tableAndOtherConditions,
            List<String> conditionColumns, int numberOfValues) {
        return "DELETE FROM " + tableAndOtherConditions + " " + buildWhereClause(conditionColumns, numberOfValues);
    }

    @Override
    public String getQueryMultipleValuesSql(String returnedValues, String tableAndFields, String valuesSql,
            int numberOfValues) {
        String values = IntStream.range(0, numberOfValues).mapToObj(i -> valuesSql).collect(Collectors.joining(", "));
        return "SELECT " + returnedValues + " FROM " + tableAndFields + " IN (" + values + ")";
    }

    @Override
    public String getQueryMultipleColumnsMultipleValuesSql(String returnedValues, String tableAndOtherFields,
            List<String> conditionColumns, int numberOfValues) {
        return "SELECT " + returnedValues + " FROM " + tableAndOtherFields + " " +
                buildWhereClause(conditionColumns, numberOfValues);
    }

    @Override
    public void executeUpdateQuery(String query, Statement statement) throws SQLException {
        log.debug("Executing query: {}", query);
        statement.executeUpdate(query);
    }

    private String buildWhereClause(List<String> conditionColumns, int numberOfValues) {
        if (numberOfValues % conditionColumns.size() != 0) {
            throw new IllegalArgumentException("numberOfValues % conditionColumns.size() should be 0");
        }
        String line = "( " + String.join(" = ? AND ", conditionColumns) + " = ? )";
        return IntStream.range(0, numberOfValues / conditionColumns.size())
                .mapToObj(i -> line)
                .collect(Collectors.joining(" OR "));
    }
}
