/*
 *
 * Artifactory is a binaries repository manager.
 * Copyright (C) 2016 JFrog Ltd.
 *
 * Artifactory is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 * Artifactory is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with Artifactory.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

package org.jfrog.storage;

import org.apache.tomcat.jdbc.pool.DataSource;
import org.apache.tomcat.jdbc.pool.PoolProperties;
import org.jfrog.storage.util.DbUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.sql.Connection;
import java.util.concurrent.TimeUnit;

/**
 * A pooling data source based on tomcat-jdbc library.
 *
 * @author Yossi Shaul
 */
public class TomcatJDBCDataSource extends DataSource implements JFrogDataSource {
    private static final Logger log = LoggerFactory.getLogger(TomcatJDBCDataSource.class);

    public TomcatJDBCDataSource(DbProperties s) {
        // see org.apache.tomcat.jdbc.pool.DataSourceFactory.parsePoolProperties()
        PoolProperties p = new PoolProperties();
        p.setUrl(s.getConnectionUrl());
        p.setDriverClassName(s.getDriverClass());
        p.setUsername(s.getUsername());
        p.setPassword(s.getPassword());

        p.setDefaultAutoCommit(s.getBooleanProperty("defaultAutoCommit", true));
        p.setDefaultTransactionIsolation(Connection.TRANSACTION_READ_COMMITTED);

        p.setInitialSize(s.getIntProperty("initialSize", 1));
        p.setMaxAge(s.getIntProperty("maxAge", 0));
        p.setMaxActive(s.getMaxActiveConnections());
        p.setMaxWait(s.getIntProperty("maxWait", (int) TimeUnit.SECONDS.toMillis(120)));
        p.setMaxIdle(s.getMaxIdleConnections());
        p.setMinIdle(s.getIntProperty("minIdle", 1));
        p.setMinEvictableIdleTimeMillis(
                s.getIntProperty("minEvictableIdleTimeMillis", 300000));
        p.setTimeBetweenEvictionRunsMillis(
                s.getIntProperty("timeBetweenEvictionRunsMillis", 30000));
        p.setInitSQL(s.getProperty("initSQL", null));

        // validation query for all kind of tests (connect, borrow etc.)
        p.setValidationQuery(s.getProperty("validationQuery", getDefaultValidationQuery(s)));
        p.setValidationQueryTimeout(s.getIntProperty("validationQueryTimeout", 30));
        p.setValidationInterval(s.getLongProperty("validationInterval", 30000));
        p.setTestOnBorrow(s.getBooleanProperty("testOnBorrow", true));
        p.setTestWhileIdle(s.getBooleanProperty("testWhileIdle", false));
        p.setTestOnReturn(s.getBooleanProperty("testOnReturn", false));
        p.setTestOnConnect(s.getBooleanProperty("testOnConnect", false));

        p.setRemoveAbandoned(s.getBooleanProperty("removeAbandoned", true));
        p.setRemoveAbandonedTimeout(s.getIntProperty("removeAbandonedTimeout", DbUtils.DEFAULT_TIMEOUT_SEC + 10));
        p.setSuspectTimeout(s.getIntProperty("suspectTimeout", 600));
        p.setLogAbandoned(s.getBooleanProperty("logAbandoned", false));
        p.setLogValidationErrors(s.getBooleanProperty("logValidationErrors", false));

        p.setJmxEnabled(s.getBooleanProperty("jmxEnabled", true));

        // only applicable if auto commit is false. has high performance penalty and only protects bugs in the code
        p.setRollbackOnReturn(s.getBooleanProperty("rollbackOnReturn", false));
        p.setCommitOnReturn(s.getBooleanProperty("commitOnReturn", false));

        p.setIgnoreExceptionOnPreLoad(s.getBooleanProperty("ignoreExceptionOnPreLoad", false));

        //p.setJdbcInterceptors(s.getProperty("jdbcInterceptors", "ConnectionState;StatementFinalizer"));
        p.setJdbcInterceptors(s.getProperty("jdbcInterceptors", null));

        p.setDefaultCatalog(s.getProperty("defaultCatalog", null));

        setPoolProperties(p);
    }

    public static JFrogDataSource createUniqueIdDataSource(DbProperties s) {
        TomcatJDBCDataSource tomcatJDBCDataSource =
                new TomcatJDBCDataSource(s);
        // see org.apache.tomcat.jdbc.pool.DataSourceFactory.parsePoolProperties()
        PoolProperties p = new PoolProperties();
        p.setName("UniqueIdDataSource");
        p.setUrl(s.getConnectionUrl());
        p.setDriverClassName(s.getDriverClass());
        p.setUsername(s.getUsername());
        p.setPassword(s.getPassword());

        // auto commit is true for the unique id generator
        p.setDefaultAutoCommit(true);
        p.setDefaultTransactionIsolation(Connection.TRANSACTION_READ_COMMITTED);

        // only one connection is required for the id generator
        p.setInitialSize(0);
        p.setMinIdle(0);
        p.setMaxIdle(1);
        p.setMaxActive(2);
        // Make sure old idle connections are sweep and tested
        long timeoutInMillis = TimeUnit.SECONDS.toMillis(s.getIntProperty("locksTimeout", 120));
        p.setTestWhileIdle(true);
        p.setTestOnBorrow(true);
        p.setTestWhileIdle(true);
        p.setRemoveAbandoned(true);
        p.setRemoveAbandonedTimeout((int) timeoutInMillis / 2);
        p.setSuspectTimeout((int) timeoutInMillis / 2);
        p.setLogAbandoned(true);
        p.setLogValidationErrors(true);

        // Timeout default to make sure new connection is created
        p.setMaxAge(timeoutInMillis);
        p.setMaxWait((int) timeoutInMillis);
        // Defaults values are good
        //p.setMinEvictableIdleTimeMillis(60000);
        //p.setTimeBetweenEvictionRunsMillis(5000);

        // Pool sweeper critical here since connection rarely used
        if (!p.isPoolSweeperEnabled()) {
            log.error("ID Generator pool connection should sweep idled connections");
        }

        // validation query for all kind of tests (connect, borrow etc.)
        p.setInitSQL(s.getProperty("initSQL", null));
        p.setValidationQuery(s.getProperty("validationQuery", getDefaultValidationQuery(s)));
        p.setValidationQueryTimeout(s.getIntProperty("validationQueryTimeout", 30));
        p.setValidationInterval(s.getLongProperty("validationInterval", 30000));

        p.setJmxEnabled(false);

        p.setIgnoreExceptionOnPreLoad(s.getBooleanProperty("ignoreExceptionOnPreLoad", false));

        //p.setJdbcInterceptors(s.getProperty("jdbcInterceptors", "ConnectionState;StatementFinalizer"));
        p.setJdbcInterceptors(s.getProperty("jdbcInterceptors", "ConnectionState"));

        p.setDefaultCatalog(s.getProperty("defaultCatalog", null));
        tomcatJDBCDataSource.setPoolProperties(p);
        return tomcatJDBCDataSource;
    }

    private static String getDefaultValidationQuery(DbProperties s) {
        switch (s.getDbType()) {
            case DERBY:
                return "values(1)";
            case MYSQL:
                // special MySQL lightweight ping query (not supported by the MariaDB connector to-date!)
                return "/* ping */";
            case ORACLE:
                return "SELECT 1 FROM DUAL";
            default:
                return "SELECT 1";
        }
    }

    @Override
    public void close() {
        close(true);    // close all connections, including active ones
    }
}
